<?php

namespace PassGram\Security;

/**
 * PGPEncryption Class
 *
 * Hybrid RSA+AES-256-GCM encryption using the user's generated key pair.
 * Instead of the shared master application key, each user's data is
 * encrypted with their own public key and can only be decrypted with
 * their private key + passphrase.
 *
 * Scheme:
 *   1. Generate a random 256-bit AES-GCM session key.
 *   2. Encrypt the plaintext with AES-256-GCM.
 *   3. Encrypt the session key with RSA-OAEP (public key).
 *   4. Bundle everything into a base64-encoded JSON envelope.
 */
class PGPEncryption
{
    /**
     * Encrypt a plaintext string with the user's PGP public key.
     *
     * @param string $plaintext    Raw data to encrypt
     * @param string $publicKeyPem PEM-encoded public key
     * @return string              Base64-encoded JSON envelope
     * @throws \Exception
     */
    public static function encrypt(string $plaintext, string $publicKeyPem): string
    {
        $publicKey = openssl_pkey_get_public($publicKeyPem);
        if ($publicKey === false) {
            throw new \Exception('PGP encryption: invalid public key – ' . openssl_error_string());
        }

        // Random 256-bit AES session key and 96-bit IV (recommended for GCM)
        $sessionKey = random_bytes(32);
        $iv         = random_bytes(12);
        $tag        = '';

        $ciphertext = openssl_encrypt(
            $plaintext,
            'aes-256-gcm',
            $sessionKey,
            OPENSSL_RAW_DATA,
            $iv,
            $tag,
            '',
            16
        );

        openssl_pkey_free($publicKey);

        if ($ciphertext === false) {
            throw new \Exception('PGP encryption: AES-GCM failed – ' . openssl_error_string());
        }

        // Wrap the session key with the user's RSA public key
        $publicKey2 = openssl_pkey_get_public($publicKeyPem);
        $encryptedSessionKey = '';
        $ok = openssl_public_encrypt(
            $sessionKey,
            $encryptedSessionKey,
            $publicKey2,
            OPENSSL_PKCS1_OAEP_PADDING
        );
        openssl_pkey_free($publicKey2);

        if (!$ok) {
            throw new \Exception('PGP encryption: RSA key wrap failed – ' . openssl_error_string());
        }

        $bundle = [
            'v'   => 1,                                      // format version
            'k'   => base64_encode($encryptedSessionKey),    // RSA-wrapped AES key
            'iv'  => base64_encode($iv),
            'tag' => base64_encode($tag),
            'd'   => base64_encode($ciphertext),
        ];

        return base64_encode(json_encode($bundle));
    }

    /**
     * Decrypt an envelope produced by encrypt().
     *
     * @param string $encrypted           Output of encrypt()
     * @param string $encryptedPrivateKey Passphrase-protected private key blob
     *                                    (same format used by PGP::generateKeyPair)
     * @param string $passphrase          PGP key passphrase
     * @return string                     Decrypted plaintext
     * @throws \Exception
     */
    public static function decrypt(
        string $encrypted,
        string $encryptedPrivateKey,
        string $passphrase
    ): string {
        $bundle = json_decode(base64_decode($encrypted), true);

        if (!$bundle || !isset($bundle['v'], $bundle['k'], $bundle['iv'], $bundle['tag'], $bundle['d'])) {
            throw new \Exception('PGP encryption: malformed envelope');
        }

        // Recover raw private key PEM from passphrase-protected blob
        $privateKeyPem = self::decryptPrivateKeyBlob($encryptedPrivateKey, $passphrase);
        $privateKey    = openssl_pkey_get_private($privateKeyPem);

        if ($privateKey === false) {
            throw new \Exception('PGP encryption: could not load private key – ' . openssl_error_string());
        }

        $sessionKey = '';
        $ok = openssl_private_decrypt(
            base64_decode($bundle['k']),
            $sessionKey,
            $privateKey,
            OPENSSL_PKCS1_OAEP_PADDING
        );
        openssl_pkey_free($privateKey);

        if (!$ok) {
            throw new \Exception('PGP encryption: RSA key unwrap failed – wrong passphrase?');
        }

        $plaintext = openssl_decrypt(
            base64_decode($bundle['d']),
            'aes-256-gcm',
            $sessionKey,
            OPENSSL_RAW_DATA,
            base64_decode($bundle['iv']),
            base64_decode($bundle['tag'])
        );

        if ($plaintext === false) {
            throw new \Exception('PGP encryption: AES-GCM decryption failed – data may be corrupt');
        }

        return $plaintext;
    }

    /**
     * JSON-encode $data and encrypt it with the user's public key.
     *
     * @param mixed  $data         Any JSON-serialisable value
     * @param string $publicKeyPem PEM public key
     * @return string Encrypted envelope
     * @throws \Exception
     */
    public static function encryptData($data, string $publicKeyPem): string
    {
        $json = json_encode($data);
        if ($json === false) {
            throw new \Exception('PGP encryption: JSON encode failed – ' . json_last_error_msg());
        }
        return self::encrypt($json, $publicKeyPem);
    }

    /**
     * Decrypt an envelope and JSON-decode the plaintext.
     *
     * @param string $encrypted           Encrypted envelope
     * @param string $encryptedPrivateKey Passphrase-protected private key blob
     * @param string $passphrase          PGP key passphrase
     * @return mixed Decoded value
     * @throws \Exception
     */
    public static function decryptData(
        string $encrypted,
        string $encryptedPrivateKey,
        string $passphrase
    ) {
        $json = self::decrypt($encrypted, $encryptedPrivateKey, $passphrase);
        $data = json_decode($json, true);
        if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception('PGP encryption: JSON decode failed – ' . json_last_error_msg());
        }
        return $data;
    }

    /**
     * Verify that $passphrase can successfully decrypt $encryptedPrivateKey.
     *
     * @param string $encryptedPrivateKey
     * @param string $passphrase
     * @return bool
     */
    public static function verifyPassphrase(string $encryptedPrivateKey, string $passphrase): bool
    {
        try {
            $pem = self::decryptPrivateKeyBlob($encryptedPrivateKey, $passphrase);
            $key = @openssl_pkey_get_private($pem);
            if ($key === false) {
                return false;
            }
            openssl_pkey_free($key);
            return true;
        } catch (\Throwable $e) {
            return false;
        }
    }

    /**
     * Decrypt the passphrase-protected private key blob.
     * The blob format is identical to what PGP::generateKeyPair() stores:
     *   base64( json({ 'encrypted': <encryptData output>, 'salt': <base64 salt> }) )
     *
     * @param string $encryptedPrivateKey
     * @param string $passphrase
     * @return string Raw PEM private key
     * @throws \Exception
     */
    private static function decryptPrivateKeyBlob(string $encryptedPrivateKey, string $passphrase): string
    {
        $data = json_decode(base64_decode($encryptedPrivateKey), true);

        if (!$data || !isset($data['encrypted'], $data['salt'])) {
            throw new \Exception('PGP encryption: invalid private key blob format');
        }

        $key       = Encryption::deriveKey($passphrase, $data['salt']);
        $keyBinary = base64_decode($key);
        $enc       = new Encryption($keyBinary);

        return $enc->decryptData($data['encrypted']);
    }
}
