<?php

namespace PassGram\Security;

/**
 * PGP Class
 *
 * Uses native PHP OpenSSL for RSA encryption operations.
 * No external dependencies required!
 */
class PGP
{
    /**
     * Generate key pair using native PHP OpenSSL
     *
     * @param string $userEmail User's email address
     * @param string $passphrase Passphrase to encrypt private key
     * @param int $keySize Key size in bits (default: 4096)
     * @param string $keyType Key algorithm: RSA, DSA, or EC (default: RSA)
     * @param string $ecCurve Elliptic curve name for EC keys (default: secp384r1)
     * @return array ['public' => string, 'private' => string, 'fingerprint' => string, 'algorithm' => string, 'key_size' => string]
     * @throws \Exception
     */
    public static function generateKeyPair(
        string $userEmail,
        string $passphrase,
        int $keySize = 4096,
        string $keyType = 'RSA',
        string $ecCurve = 'secp384r1'
    ): array
    {
        // Configure key generation based on algorithm type
        $config = [];
        $algorithm = '';
        $keySizeDisplay = '';

        switch (strtoupper($keyType)) {
            case 'RSA':
                $config = [
                    'private_key_bits' => $keySize,
                    'private_key_type' => OPENSSL_KEYTYPE_RSA,
                ];
                $algorithm = 'RSA';
                $keySizeDisplay = $keySize . ' bits';
                break;

            case 'DSA':
                $config = [
                    'private_key_bits' => $keySize,
                    'private_key_type' => OPENSSL_KEYTYPE_DSA,
                ];
                $algorithm = 'DSA';
                $keySizeDisplay = $keySize . ' bits';
                break;

            case 'EC':
                $config = [
                    'private_key_type' => OPENSSL_KEYTYPE_EC,
                    'curve_name' => $ecCurve,
                ];
                $algorithm = 'EC';
                $keySizeDisplay = $ecCurve;
                break;

            default:
                throw new \Exception('Invalid key type. Supported types: RSA, DSA, EC');
        }

        // Generate private key
        $privateKeyResource = openssl_pkey_new($config);

        if ($privateKeyResource === false) {
            throw new \Exception('Failed to generate ' . $algorithm . ' key pair: ' . openssl_error_string());
        }

        // Export private key
        $privateKeyString = '';
        openssl_pkey_export($privateKeyResource, $privateKeyString);

        // Get public key
        $publicKeyDetails = openssl_pkey_get_details($privateKeyResource);
        $publicKeyString = $publicKeyDetails['key'];

        // Encrypt private key with passphrase
        $encryptedPrivate = self::encryptPrivateKey($privateKeyString, $passphrase);

        // Calculate fingerprint
        $fingerprint = self::calculateFingerprint($publicKeyString);

        // Free the key resource
        openssl_pkey_free($privateKeyResource);

        return [
            'public' => $publicKeyString,
            'private' => $encryptedPrivate,
            'fingerprint' => $fingerprint,
            'algorithm' => $algorithm,
            'key_size' => $keySizeDisplay,
        ];
    }

    /**
     * Encrypt data with public key
     *
     * @param string $plaintext Data to encrypt
     * @param string $publicKeyString Public key (PEM format)
     * @return string Base64-encoded encrypted data
     * @throws \Exception
     */
    public static function encrypt(string $plaintext, string $publicKeyString): string
    {
        $publicKey = openssl_pkey_get_public($publicKeyString);

        if ($publicKey === false) {
            throw new \Exception('Invalid public key: ' . openssl_error_string());
        }

        $encrypted = '';
        $result = openssl_public_encrypt($plaintext, $encrypted, $publicKey, OPENSSL_PKCS1_OAEP_PADDING);

        openssl_pkey_free($publicKey);

        if (!$result) {
            throw new \Exception('Encryption failed: ' . openssl_error_string());
        }

        return base64_encode($encrypted);
    }

    /**
     * Decrypt data with private key
     *
     * @param string $ciphertext Base64-encoded encrypted data
     * @param string $encryptedPrivateKey Encrypted private key
     * @param string $passphrase Passphrase to decrypt private key
     * @return string Decrypted plaintext
     * @throws \Exception
     */
    public static function decrypt(string $ciphertext, string $encryptedPrivateKey, string $passphrase): string
    {
        // Decrypt private key
        $privateKeyString = self::decryptPrivateKey($encryptedPrivateKey, $passphrase);

        // Load private key
        $privateKey = openssl_pkey_get_private($privateKeyString);

        if ($privateKey === false) {
            throw new \Exception('Invalid private key: ' . openssl_error_string());
        }

        // Decode ciphertext
        $encrypted = base64_decode($ciphertext);

        $decrypted = '';
        $result = openssl_private_decrypt($encrypted, $decrypted, $privateKey, OPENSSL_PKCS1_OAEP_PADDING);

        openssl_pkey_free($privateKey);

        if (!$result) {
            throw new \Exception('Decryption failed: ' . openssl_error_string());
        }

        return $decrypted;
    }

    /**
     * Encrypt private key with passphrase
     *
     * @param string $privateKey Private key string
     * @param string $passphrase Passphrase
     * @return string Encrypted private key (base64-encoded JSON)
     * @throws \Exception
     */
    private static function encryptPrivateKey(string $privateKey, string $passphrase): string
    {
        // Use AES-256-GCM to encrypt the private key
        $salt = Encryption::generateSalt();
        $key = Encryption::deriveKey($passphrase, $salt);

        // deriveKey returns base64-encoded key, decode it for encryption
        $keyBinary = base64_decode($key);

        $encryption = new Encryption($keyBinary);
        $encrypted = $encryption->encryptData($privateKey);

        return base64_encode(json_encode([
            'encrypted' => $encrypted,
            'salt' => $salt,
        ]));
    }

    /**
     * Decrypt private key with passphrase
     *
     * @param string $encryptedPrivateKey Encrypted private key
     * @param string $passphrase Passphrase
     * @return string Decrypted private key
     * @throws \Exception
     */
    private static function decryptPrivateKey(string $encryptedPrivateKey, string $passphrase): string
    {
        $data = json_decode(base64_decode($encryptedPrivateKey), true);

        if (!$data || !isset($data['encrypted']) || !isset($data['salt'])) {
            throw new \Exception('Invalid encrypted private key format');
        }

        $key = Encryption::deriveKey($passphrase, $data['salt']);

        // deriveKey returns base64-encoded key, decode it for encryption
        $keyBinary = base64_decode($key);

        $encryption = new Encryption($keyBinary);

        return $encryption->decryptData($data['encrypted']);
    }

    /**
     * Calculate fingerprint of public key
     *
     * @param string $publicKey Public key string
     * @return string Fingerprint (hex)
     */
    private static function calculateFingerprint(string $publicKey): string
    {
        return hash('sha256', $publicKey);
    }

    /**
     * Validate public key format.
     *
     * Accepts PEM/DER (X.509, PKCS#8), OpenPGP ASCII-armored blocks,
     * SSH public keys (ssh-rsa, ssh-ed25519, ecdsa-*, etc.),
     * PuTTY key files, and OpenSSH private key blocks.
     *
     * @param string $publicKey Public key to validate
     * @return bool
     */
    public static function validatePublicKey(string $publicKey): bool
    {
        $trimmed = trim($publicKey);
        if ($trimmed === '') {
            return false;
        }

        // --- Detect non-OpenSSL formats FIRST to avoid triggering
        //     openssl_pkey_get_public() on content it cannot parse.
        //     In PHP 8.x that call emits E_WARNING, which may be converted
        //     to an exception by the application's error handler. ---

        // OpenPGP ASCII-armored public key block (.asc / .eml / .gpg)
        if (strpos($trimmed, '-----BEGIN PGP PUBLIC KEY BLOCK-----') !== false) {
            return true;
        }

        // OpenSSH private key block
        if (strpos($trimmed, '-----BEGIN OPENSSH PRIVATE KEY-----') !== false) {
            return true;
        }

        // SSH public key formats (one-liner: type base64 [comment])
        $sshTypes = [
            'ssh-rsa', 'ssh-dss', 'ssh-ed25519',
            'ecdsa-sha2-nistp256', 'ecdsa-sha2-nistp384', 'ecdsa-sha2-nistp521',
            'sk-ssh-ed25519@openssh.com', 'sk-ecdsa-sha2-nistp256@openssh.com',
            'ssh-xmss@openssh.com',
        ];
        // Check each line (authorized_keys files may have multiple entries)
        foreach (explode("\n", $trimmed) as $line) {
            $line = ltrim($line);
            foreach ($sshTypes as $t) {
                if (strncmp($line, $t . ' ', strlen($t) + 1) === 0) {
                    return true;
                }
            }
        }

        // PuTTY key file (.ppk) – versions 2 and 3
        if (preg_match('/^PuTTY-User-Key-File-[23]:/m', $trimmed)) {
            return true;
        }

        // OpenSSL-parseable PEM/DER (X.509, PKCS#8, PKCS#1 …).
        // Use @ to suppress E_WARNING on PHP 8 for unrecognised content.
        $key = @openssl_pkey_get_public($trimmed);
        if ($key !== false) {
            openssl_pkey_free($key);
            return true;
        }

        // Generic -----BEGIN … ----- block catch-all
        // (handles CERTIFICATE, RSA PUBLIC KEY, EC PUBLIC KEY, etc.)
        if (preg_match('/-----BEGIN [A-Z0-9 ]+----- *\r?\n/', $trimmed)) {
            return true;
        }

        return false;
    }

    /**
     * Export public key (same as input, but validated)
     *
     * @param string $publicKey Public key
     * @return string
     * @throws \Exception
     */
    public static function exportPublicKey(string $publicKey): string
    {
        if (!self::validatePublicKey($publicKey)) {
            throw new \Exception('Invalid public key');
        }

        return $publicKey;
    }

    /**
     * Get key info
     *
     * Supports PEM/DER (X.509, PKCS#8), OpenPGP ASCII-armored blocks,
     * SSH public keys, PuTTY key files, and OpenSSH blocks.
     *
     * @param string $publicKey Public key
     * @return array Key information (type, bits, fingerprint, details)
     * @throws \Exception
     */
    public static function getKeyInfo(string $publicKey): array
    {
        $trimmed = trim($publicKey);

        // Non-OpenSSL formats are detected before calling openssl_pkey_get_public()
        // to avoid PHP 8 E_WARNING on unrecognised content.

        // OpenPGP ASCII-armored public key block (.asc / .eml / .gpg)
        if (strpos($trimmed, '-----BEGIN PGP PUBLIC KEY BLOCK-----') !== false) {
            return [
                'type'        => 'OpenPGP',
                'bits'        => 0,
                'fingerprint' => self::calculateFingerprint($publicKey),
                'details'     => [],
            ];
        }

        // OpenSSH private key block
        if (strpos($trimmed, '-----BEGIN OPENSSH PRIVATE KEY-----') !== false) {
            return [
                'type'        => 'OpenSSH',
                'bits'        => 0,
                'fingerprint' => self::calculateFingerprint($publicKey),
                'details'     => [],
            ];
        }

        // SSH public key – detect algorithm from key-type token
        $sshTypeMap = [
            'ssh-rsa'                            => 'SSH/RSA',
            'ssh-dss'                            => 'SSH/DSA',
            'ssh-ed25519'                        => 'SSH/Ed25519',
            'ecdsa-sha2-nistp256'                => 'SSH/ECDSA-256',
            'ecdsa-sha2-nistp384'                => 'SSH/ECDSA-384',
            'ecdsa-sha2-nistp521'                => 'SSH/ECDSA-521',
            'sk-ssh-ed25519@openssh.com'         => 'SSH/Ed25519-SK',
            'sk-ecdsa-sha2-nistp256@openssh.com' => 'SSH/ECDSA-SK',
        ];
        foreach (explode("\n", $trimmed) as $line) {
            $line = ltrim($line);
            foreach ($sshTypeMap as $token => $label) {
                if (strncmp($line, $token . ' ', strlen($token) + 1) === 0) {
                    return [
                        'type'        => $label,
                        'bits'        => 0,
                        'fingerprint' => self::calculateFingerprint($publicKey),
                        'details'     => [],
                    ];
                }
            }
        }

        // PuTTY key file (.ppk)
        if (preg_match('/^PuTTY-User-Key-File-[23]:\s*(\S+)/m', $trimmed, $m)) {
            return [
                'type'        => 'PuTTY/' . $m[1],
                'bits'        => 0,
                'fingerprint' => self::calculateFingerprint($publicKey),
                'details'     => [],
            ];
        }

        // Try OpenSSL for PEM/DER (X.509, PKCS#8, PKCS#1 …).
        // @ suppresses E_WARNING on PHP 8 for unrecognised content.
        $key = @openssl_pkey_get_public($trimmed);
        if ($key !== false) {
            $details = openssl_pkey_get_details($key);
            openssl_pkey_free($key);

            $type = 'Unknown';
            switch ($details['type']) {
                case OPENSSL_KEYTYPE_RSA: $type = 'RSA'; break;
                case OPENSSL_KEYTYPE_DSA: $type = 'DSA'; break;
                case OPENSSL_KEYTYPE_EC:  $type = 'EC';  break;
                case OPENSSL_KEYTYPE_DH:  $type = 'DH';  break;
            }

            return [
                'type'        => $type,
                'bits'        => $details['bits'],
                'fingerprint' => self::calculateFingerprint($publicKey),
                'details'     => $details,
            ];
        }

        // Generic -----BEGIN … ----- block catch-all
        if (preg_match('/-----BEGIN ([A-Z0-9 ]+)-----/', $trimmed, $m)) {
            return [
                'type'        => trim($m[1]),
                'bits'        => 0,
                'fingerprint' => self::calculateFingerprint($publicKey),
                'details'     => [],
            ];
        }

        throw new \Exception('Unrecognised key format');
    }

    /**
     * Sign data with private key
     *
     * @param string $data Data to sign
     * @param string $encryptedPrivateKey Encrypted private key
     * @param string $passphrase Passphrase
     * @return string Base64-encoded signature
     * @throws \Exception
     */
    public static function sign(string $data, string $encryptedPrivateKey, string $passphrase): string
    {
        $privateKeyString = self::decryptPrivateKey($encryptedPrivateKey, $passphrase);
        $privateKey = openssl_pkey_get_private($privateKeyString);

        if ($privateKey === false) {
            throw new \Exception('Invalid private key');
        }

        $signature = '';
        $result = openssl_sign($data, $signature, $privateKey, OPENSSL_ALGO_SHA256);

        openssl_pkey_free($privateKey);

        if (!$result) {
            throw new \Exception('Signing failed: ' . openssl_error_string());
        }

        return base64_encode($signature);
    }

    /**
     * Verify signature with public key
     *
     * @param string $data Original data
     * @param string $signature Base64-encoded signature
     * @param string $publicKey Public key
     * @return bool True if signature is valid
     * @throws \Exception
     */
    public static function verify(string $data, string $signature, string $publicKey): bool
    {
        $key = openssl_pkey_get_public($publicKey);

        if ($key === false) {
            throw new \Exception('Invalid public key');
        }

        $signatureBinary = base64_decode($signature);
        $result = openssl_verify($data, $signatureBinary, $key, OPENSSL_ALGO_SHA256);

        openssl_pkey_free($key);

        return $result === 1;
    }
}
