<?php

namespace PassGram\Security;

/**
 * Encryption Class
 *
 * Handles AES-256-GCM encryption and decryption for PassGram.
 * Uses OpenSSL for encryption operations and Argon2id for key derivation.
 */
class Encryption
{
    private string $masterKey;
    private array $config;

    /**
     * Constructor
     *
     * @param string $masterKey The master application key
     * @param array $config Encryption configuration
     */
    public function __construct(string $masterKey, array $config = [])
    {
        $this->masterKey = $masterKey;
        $this->config = array_merge([
            'algorithm' => 'aes-256-gcm',
            'iv_length' => 12,
            'tag_length' => 16,
            'key_length' => 32,
        ], $config);
    }

    /**
     * Encrypt data using AES-256-GCM
     *
     * @param string $plaintext The data to encrypt
     * @param string|null $key Optional key (uses master key if not provided)
     * @return array ['ciphertext' => string, 'iv' => string, 'tag' => string]
     * @throws \Exception
     */
    public function encrypt(string $plaintext, ?string $key = null): array
    {
        $key = $key ?? $this->masterKey;

        // Validate key length
        if (strlen($key) !== $this->config['key_length']) {
            throw new \Exception('Invalid key length');
        }

        // Generate random IV
        $iv = $this->generateIV();

        // Initialize tag variable
        $tag = '';

        // Encrypt the data
        $ciphertext = openssl_encrypt(
            $plaintext,
            $this->config['algorithm'],
            $key,
            OPENSSL_RAW_DATA,
            $iv,
            $tag,
            '',
            $this->config['tag_length']
        );

        if ($ciphertext === false) {
            throw new \Exception('Encryption failed: ' . openssl_error_string());
        }

        return [
            'ciphertext' => base64_encode($ciphertext),
            'iv' => base64_encode($iv),
            'tag' => base64_encode($tag),
        ];
    }

    /**
     * Decrypt data using AES-256-GCM
     *
     * @param string $ciphertext Base64-encoded ciphertext
     * @param string $iv Base64-encoded IV
     * @param string $tag Base64-encoded authentication tag
     * @param string|null $key Optional key (uses master key if not provided)
     * @return string Decrypted plaintext
     * @throws \Exception
     */
    public function decrypt(string $ciphertext, string $iv, string $tag, ?string $key = null): string
    {
        $key = $key ?? $this->masterKey;

        // Validate key length
        if (strlen($key) !== $this->config['key_length']) {
            throw new \Exception('Invalid key length');
        }

        // Decode base64 values
        $ciphertext = base64_decode($ciphertext);
        $iv = base64_decode($iv);
        $tag = base64_decode($tag);

        // Decrypt the data
        $plaintext = openssl_decrypt(
            $ciphertext,
            $this->config['algorithm'],
            $key,
            OPENSSL_RAW_DATA,
            $iv,
            $tag
        );

        if ($plaintext === false) {
            throw new \Exception('Decryption failed: ' . openssl_error_string());
        }

        return $plaintext;
    }

    /**
     * Encrypt a JSON-encodable structure
     *
     * @param mixed $data Data to encrypt (will be JSON encoded)
     * @param string|null $key Optional key
     * @return string Base64-encoded encrypted package (ciphertext:iv:tag)
     * @throws \Exception
     */
    public function encryptData($data, ?string $key = null): string
    {
        $json = json_encode($data);
        if ($json === false) {
            $errorMsg = json_last_error_msg();
            $dataType = gettype($data);
            throw new \Exception('JSON encoding failed: ' . $errorMsg . ' (Data type: ' . $dataType . ')');
        }

        $encrypted = $this->encrypt($json, $key);

        // Combine into single string for storage
        return $encrypted['ciphertext'] . ':' . $encrypted['iv'] . ':' . $encrypted['tag'];
    }

    /**
     * Decrypt a JSON-encoded structure
     *
     * @param string $encryptedPackage Encrypted package (ciphertext:iv:tag)
     * @param string|null $key Optional key
     * @param bool $assoc Return as associative array
     * @return mixed Decrypted and decoded data
     * @throws \Exception
     */
    public function decryptData(string $encryptedPackage, ?string $key = null, bool $assoc = true)
    {
        // Split the package
        $parts = explode(':', $encryptedPackage);
        if (count($parts) !== 3) {
            throw new \Exception('Invalid encrypted package format');
        }

        [$ciphertext, $iv, $tag] = $parts;

        $json = $this->decrypt($ciphertext, $iv, $tag, $key);

        $data = json_decode($json, $assoc);
        if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception('JSON decoding failed: ' . json_last_error_msg());
        }

        return $data;
    }

    /**
     * Generate a random encryption key
     *
     * @return string Random key (32 bytes)
     * @throws \Exception
     */
    public static function generateKey(): string
    {
        $key = random_bytes(32);
        if ($key === false) {
            throw new \Exception('Failed to generate random key');
        }
        return $key;
    }

    /**
     * Generate a random IV
     *
     * @return string Random IV
     * @throws \Exception
     */
    private function generateIV(): string
    {
        $iv = random_bytes($this->config['iv_length']);
        if ($iv === false) {
            throw new \Exception('Failed to generate random IV');
        }
        return $iv;
    }

    /**
     * Derive encryption key from password using Argon2id
     *
     * @param string $password User's password
     * @param string $salt Salt for key derivation
     * @param array $options Argon2id options
     * @return string Derived key (32 bytes)
     * @throws \Exception
     */
    public static function deriveKey(string $password, string $salt, array $options = []): string
    {
        $defaults = [
            'memory_cost' => 65536, // 64 MB
            'time_cost' => 4,
            'threads' => 2,
        ];

        $options = array_merge($defaults, $options);

        $hash = password_hash(
            $password,
            PASSWORD_ARGON2ID,
            [
                'memory_cost' => $options['memory_cost'],
                'time_cost' => $options['time_cost'],
                'threads' => $options['threads'],
            ]
        );

        if ($hash === false) {
            throw new \Exception('Key derivation failed');
        }

        // Extract 32 bytes for the key
        // Hash password with salt to get consistent key
        $key = hash_pbkdf2('sha256', $password, $salt, 100000, 32, true);

        // Return base64-encoded key for JSON compatibility
        return base64_encode($key);
    }

    /**
     * Generate a random salt
     *
     * @return string Random salt (32 bytes, base64 encoded)
     * @throws \Exception
     */
    public static function generateSalt(): string
    {
        $salt = random_bytes(32);
        if ($salt === false) {
            throw new \Exception('Failed to generate random salt');
        }
        return base64_encode($salt);
    }

    /**
     * Hash a password using bcrypt (for authentication)
     *
     * @param string $password Password to hash
     * @param int $cost Bcrypt cost factor
     * @return string Hashed password
     * @throws \Exception
     */
    public static function hashPassword(string $password, int $cost = 12): string
    {
        $hash = password_hash($password, PASSWORD_BCRYPT, ['cost' => $cost]);
        if ($hash === false) {
            throw new \Exception('Password hashing failed');
        }
        return $hash;
    }

    /**
     * Verify a password against a hash
     *
     * @param string $password Password to verify
     * @param string $hash Hash to verify against
     * @return bool True if password matches
     */
    public static function verifyPassword(string $password, string $hash): bool
    {
        return password_verify($password, $hash);
    }

    /**
     * Check if a password hash needs rehashing
     *
     * @param string $hash The hash to check
     * @param int $cost The desired cost factor
     * @return bool True if rehashing is needed
     */
    public static function needsRehash(string $hash, int $cost = 12): bool
    {
        return password_needs_rehash($hash, PASSWORD_BCRYPT, ['cost' => $cost]);
    }

    /**
     * Generate a random token
     *
     * @param int $length Length in bytes
     * @return string Random token (hex encoded)
     * @throws \Exception
     */
    public static function generateToken(int $length = 32): string
    {
        $token = random_bytes($length);
        if ($token === false) {
            throw new \Exception('Failed to generate random token');
        }
        return bin2hex($token);
    }
}
