<?php

namespace PassGram\Security;

use PassGram\Core\Session;
use PassGram\Core\Config;

/**
 * CSRF Protection Class
 *
 * Generates and validates CSRF tokens to prevent Cross-Site Request Forgery attacks.
 */
class CSRF
{
    private string $tokenKey = '_csrf_token';

    public function __construct()
    {
    }

    /**
     * Generate CSRF token
     *
     * @return string
     * @throws \Exception
     */
    public function generateToken(): string
    {
        $token = Encryption::generateToken(32);

        Session::set($this->tokenKey, [
            'token' => $token,
            'timestamp' => time(),
        ]);

        return $token;
    }

    /**
     * Get current CSRF token (generate if doesn't exist)
     *
     * @return string
     * @throws \Exception
     */
    public function getToken(): string
    {
        $tokenData = Session::get($this->tokenKey);

        if (!$tokenData || !isset($tokenData['token'])) {
            return $this->generateToken();
        }

        // Check if token has expired (1 hour default)
        $lifetime = 3600;
        if (time() - $tokenData['timestamp'] > $lifetime) {
            return $this->generateToken();
        }

        return $tokenData['token'];
    }

    /**
     * Validate CSRF token
     *
     * @param string $token Token to validate
     * @return bool
     */
    public function validateToken(string $token): bool
    {
        $tokenData = Session::get($this->tokenKey);

        if (!$tokenData || !isset($tokenData['token'])) {
            return false;
        }

        // Check if token has expired (1 hour default)
        $lifetime = 3600;
        if (time() - $tokenData['timestamp'] > $lifetime) {
            return false;
        }

        // Timing-safe comparison
        return hash_equals($tokenData['token'], $token);
    }

    /**
     * Validate CSRF token from POST data
     *
     * @return bool
     */
    public function validatePost(): bool
    {
        $token = $_POST['csrf_token'] ?? '';
        return $this->validateToken($token);
    }

    /**
     * Get HTML hidden input field with CSRF token
     *
     * @return string
     * @throws \Exception
     */
    public function getTokenField(): string
    {
        $token = htmlspecialchars($this->getToken(), ENT_QUOTES, 'UTF-8');
        return '<input type="hidden" name="csrf_token" value="' . $token . '">';
    }

    /**
     * Rotate CSRF token
     *
     * @return string New token
     * @throws \Exception
     */
    public function rotateToken(): string
    {
        return $this->generateToken();
    }
}
