<?php

namespace PassGram\Security;

use PassGram\Core\Session;
use PassGram\Core\Database;
use PassGram\Core\Config;
use PassGram\Helpers\Logger;

/**
 * Authentication Class
 *
 * Handles user authentication and authorization for PassGram.
 */
class Auth
{
    private Database $db;
    private Config $config;
    private Logger $logger;

    public function __construct(Database $db, Config $config, Logger $logger)
    {
        $this->db = $db;
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * Attempt to log in a user
     *
     * @param string $username Username
     * @param string $password Password
     * @return array|null User data if successful, null otherwise
     * @throws \Exception
     */
    public function login(string $username, string $password): ?array
    {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

        // Check rate limiting
        if (!$this->checkRateLimit($username, $ip)) {
            $this->logger->authAttempt($username, false, $ip);
            throw new \Exception('Too many login attempts. Please try again later.');
        }

        // Find user
        $user = $this->findUserByUsername($username);

        if (!$user) {
            $this->recordFailedAttempt($username, $ip);
            $this->logger->authAttempt($username, false, $ip);
            return null;
        }

        // Verify password
        if (!Encryption::verifyPassword($password, $user['password_hash'])) {
            $this->recordFailedAttempt($username, $ip);
            $this->logger->authAttempt($username, false, $ip);
            return null;
        }

        // Check if account is active
        if (isset($user['status']) && $user['status'] !== 'active') {
            $this->logger->authAttempt($username, false, $ip);
            throw new \Exception('Account is suspended');
        }

        // Successful login
        $this->clearFailedAttempts($username, $ip);
        $this->createSession($user);
        $this->updateLastLogin($user['id']);

        $this->logger->authAttempt($username, true, $ip);

        return $user;
    }

    /**
     * Log out current user
     *
     * @return bool
     */
    public function logout(): bool
    {
        return Session::destroy();
    }

    /**
     * Check if user is authenticated
     *
     * @return bool
     */
    public function check(): bool
    {
        return Session::has('user_id') && Session::has('logged_in_at');
    }

    /**
     * Check if user is authenticated (alias for check())
     *
     * @return bool
     */
    public function isLoggedIn(): bool
    {
        return $this->check();
    }

    /**
     * Get current user ID
     *
     * @return string|null
     */
    public function getUserId(): ?string
    {
        return Session::get('user_id');
    }

    /**
     * Get current user data
     *
     * @return array|null
     * @throws \Exception
     */
    public function getCurrentUser(): ?array
    {
        $userId = $this->getUserId();

        if (!$userId) {
            return null;
        }

        return $this->findUserById($userId);
    }

    /**
     * Require authentication (redirect if not logged in)
     *
     * @param string $redirectUrl URL to redirect to if not authenticated
     * @return void
     */
    public function requireAuth(string $redirectUrl = '/login.php'): void
    {
        if (!$this->isLoggedIn()) {
            header('Location: ' . $redirectUrl);
            exit;
        }
    }

    /**
     * Check if user has permission
     *
     * @param string $permission Permission to check
     * @return bool
     * @throws \Exception
     */
    public function hasPermission(string $permission): bool
    {
        $user = $this->getCurrentUser();

        if (!$user) {
            return false;
        }

        // Implement permission logic here
        // For now, return true for authenticated users
        return true;
    }

    /**
     * Create session for user
     *
     * @param array $user User data
     * @return void
     */
    private function createSession(array $user): void
    {
        Session::regenerate();
        Session::set('user_id', $user['id']);
        Session::set('username', $user['username']);
        Session::set('logged_in_at', time());
        Session::set('last_activity', time());
        Session::set('ip_address', $_SERVER['REMOTE_ADDR'] ?? '');
        Session::set('user_agent', $_SERVER['HTTP_USER_AGENT'] ?? '');
    }

    /**
     * Find user by username
     *
     * @param string $username Username
     * @return array|null
     * @throws \Exception
     */
    private function findUserByUsername(string $username): ?array
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as $user) {
            if ($user['username'] === $username) {
                return $user;
            }
        }

        return null;
    }

    /**
     * Find user by ID
     *
     * @param string $userId User ID
     * @return array|null
     * @throws \Exception
     */
    private function findUserById(string $userId): ?array
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as $user) {
            if ($user['id'] === $userId) {
                return $user;
            }
        }

        return null;
    }

    /**
     * Update user's last login timestamp
     *
     * @param string $userId User ID
     * @return void
     * @throws \Exception
     */
    private function updateLastLogin(string $userId): void
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $userId) {
                $user['last_login'] = date('c');
                break;
            }
        }

        $this->db->writeUsers($users);
    }

    /**
     * Check rate limiting for login attempts
     *
     * @param string $username Username
     * @param string $ip IP address
     * @return bool True if within limits
     */
    private function checkRateLimit(string $username, string $ip): bool
    {
        $key = 'login_attempts_' . md5($username . $ip);
        $attempts = Session::get($key, []);

        $maxAttempts = $this->config->get('security.login_rate_limit', 5);
        $timeframe = $this->config->get('security.login_rate_timeframe', 900);

        // Clean old attempts
        $attempts = array_filter($attempts, function ($timestamp) use ($timeframe) {
            return (time() - $timestamp) < $timeframe;
        });

        return count($attempts) < $maxAttempts;
    }

    /**
     * Record failed login attempt
     *
     * @param string $username Username
     * @param string $ip IP address
     * @return void
     */
    private function recordFailedAttempt(string $username, string $ip): void
    {
        $key = 'login_attempts_' . md5($username . $ip);
        $attempts = Session::get($key, []);
        $attempts[] = time();
        Session::set($key, $attempts);
    }

    /**
     * Clear failed login attempts
     *
     * @param string $username Username
     * @param string $ip IP address
     * @return void
     */
    private function clearFailedAttempts(string $username, string $ip): void
    {
        $key = 'login_attempts_' . md5($username . $ip);
        Session::remove($key);
    }
}
