<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Core\Config;
use PassGram\Security\Encryption;
use PassGram\Helpers\Validator;

/**
 * User Model
 *
 * Handles user CRUD operations and user-related business logic.
 */
class User
{
    private Database $db;
    private Config $config;
    private Validator $validator;

    public function __construct(Database $db, Config $config, Validator $validator)
    {
        $this->db = $db;
        $this->config = $config;
        $this->validator = $validator;
    }

    /**
     * Create a new user
     *
     * @param array $data User data
     * @return array Created user
     * @throws \Exception
     */
    public function create(array $data): array
    {
        // Validate required fields
        if (!$this->validator->required($data['username'] ?? '', 'username')) {
            throw new \Exception($this->validator->getFirstError());
        }

        if (!$this->validator->username($data['username'])) {
            throw new \Exception($this->validator->getFirstError());
        }

        if (!$this->validator->email($data['email'] ?? '')) {
            throw new \Exception($this->validator->getFirstError());
        }

        if (!$this->validator->password($data['password'] ?? '')) {
            throw new \Exception($this->validator->getFirstError());
        }

        // Check if username or email already exists
        if ($this->findByUsername($data['username'])) {
            throw new \Exception('Username already exists');
        }

        if ($this->findByEmail($data['email'])) {
            throw new \Exception('Email already exists');
        }

        // Create user object
        $user = [
            'id' => Validator::generateUUID(),
            'username' => $data['username'],
            'email' => $data['email'],
            'password_hash' => Encryption::hashPassword($data['password']),
            'master_key_hash' => $this->generateMasterKeyHash($data['password']),
            'created_at' => date('c'),
            'updated_at' => date('c'),
            'last_login' => null,
            'group_ids' => $data['group_ids'] ?? [],
            'has_pgp_key' => false,
            'pgp_fingerprint' => null,
            'status' => 'active',
            'settings' => [
                'session_timeout' => 3600,
                'two_factor_enabled' => false,
            ],
        ];

        // Add to database
        $users = $this->db->readUsers();
        $users['users'][] = $user;
        $this->db->writeUsers($users);

        // Remove password from return value
        unset($user['password_hash']);
        unset($user['master_key_hash']);

        return $user;
    }

    /**
     * Find user by ID
     *
     * @param string $id User ID
     * @return array|null
     * @throws \Exception
     */
    public function findById(string $id): ?array
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as $user) {
            if ($user['id'] === $id) {
                return $user;
            }
        }

        return null;
    }

    /**
     * Find user by username
     *
     * @param string $username Username
     * @return array|null
     * @throws \Exception
     */
    public function findByUsername(string $username): ?array
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as $user) {
            if ($user['username'] === $username) {
                return $user;
            }
        }

        return null;
    }

    /**
     * Find user by email
     *
     * @param string $email Email
     * @return array|null
     * @throws \Exception
     */
    public function findByEmail(string $email): ?array
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as $user) {
            if ($user['email'] === $email) {
                return $user;
            }
        }

        return null;
    }

    /**
     * Update user
     *
     * @param string $id User ID
     * @param array $data Data to update
     * @return array Updated user
     * @throws \Exception
     */
    public function update(string $id, array $data): array
    {
        $users = $this->db->readUsers();
        $updated = null;

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $id) {
                // Update allowed fields
                if (isset($data['email']) && $data['email'] !== $user['email']) {
                    if (!$this->validator->email($data['email'])) {
                        throw new \Exception($this->validator->getFirstError());
                    }
                    $user['email'] = $data['email'];
                }

                if (isset($data['status'])) {
                    $user['status'] = $data['status'];
                }

                if (isset($data['settings'])) {
                    $user['settings'] = array_merge($user['settings'], $data['settings']);
                }

                $user['updated_at'] = date('c');
                $updated = $user;
                break;
            }
        }

        if (!$updated) {
            throw new \Exception('User not found');
        }

        $this->db->writeUsers($users);

        return $updated;
    }

    /**
     * Delete user
     *
     * @param string $id User ID
     * @return bool
     * @throws \Exception
     */
    public function delete(string $id): bool
    {
        $users = $this->db->readUsers();
        $found = false;

        $users['users'] = array_filter($users['users'], function ($user) use ($id, &$found) {
            if ($user['id'] === $id) {
                $found = true;
                return false;
            }
            return true;
        });

        $users['users'] = array_values($users['users']);

        if (!$found) {
            throw new \Exception('User not found');
        }

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Update user password
     *
     * @param string $id User ID
     * @param string $newPassword New password
     * @return bool
     * @throws \Exception
     */
    public function updatePassword(string $id, string $newPassword): bool
    {
        if (!$this->validator->password($newPassword)) {
            throw new \Exception($this->validator->getFirstError());
        }

        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $id) {
                $user['password_hash'] = Encryption::hashPassword($newPassword);
                $user['master_key_hash'] = $this->generateMasterKeyHash($newPassword);
                $user['updated_at'] = date('c');
                break;
            }
        }

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Add user to group
     *
     * @param string $userId User ID
     * @param string $groupId Group ID
     * @return bool
     * @throws \Exception
     */
    public function addToGroup(string $userId, string $groupId): bool
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $userId) {
                if (!in_array($groupId, $user['group_ids'])) {
                    $user['group_ids'][] = $groupId;
                    $user['updated_at'] = date('c');
                }
                break;
            }
        }

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Remove user from group
     *
     * @param string $userId User ID
     * @param string $groupId Group ID
     * @return bool
     * @throws \Exception
     */
    public function removeFromGroup(string $userId, string $groupId): bool
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $userId) {
                $user['group_ids'] = array_filter($user['group_ids'], function ($gid) use ($groupId) {
                    return $gid !== $groupId;
                });
                $user['group_ids'] = array_values($user['group_ids']);
                $user['updated_at'] = date('c');
                break;
            }
        }

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Set PGP key fingerprint
     *
     * @param string $userId User ID
     * @param string $fingerprint PGP key fingerprint
     * @return bool
     * @throws \Exception
     */
    public function setPGPFingerprint(string $userId, string $fingerprint): bool
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $userId) {
                $user['has_pgp_key'] = true;
                $user['pgp_fingerprint'] = $fingerprint;
                $user['updated_at'] = date('c');
                break;
            }
        }

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Set PGP key information (fingerprint, algorithm, key size)
     *
     * @param string $userId User ID
     * @param string $fingerprint PGP key fingerprint
     * @param string $algorithm Key algorithm (RSA, DSA, EC)
     * @param string $keySize Key size display (e.g., "4096 bits" or "secp384r1")
     * @return bool
     * @throws \Exception
     */
    public function setPGPKeyInfo(string $userId, string $fingerprint, string $algorithm, string $keySize): bool
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $userId) {
                $user['has_pgp_key'] = true;
                $user['pgp_fingerprint'] = $fingerprint;
                $user['pgp_algorithm'] = $algorithm;
                $user['pgp_key_size'] = $keySize;
                $user['updated_at'] = date('c');
                break;
            }
        }

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Enable or disable PGP encryption mode for a user.
     *
     * When enabled, credentials and public-key stores are encrypted with
     * the user's own PGP key pair instead of the master application key.
     *
     * @param string $userId  User ID
     * @param bool   $enabled True to enable PGP mode, false to revert to AES mode
     * @return bool
     * @throws \Exception
     */
    public function setPGPEncryptionMode(string $userId, bool $enabled): bool
    {
        $users = $this->db->readUsers();

        foreach ($users['users'] as &$user) {
            if ($user['id'] === $userId) {
                if (!isset($user['settings'])) {
                    $user['settings'] = [];
                }
                $user['settings']['pgp_encryption_mode'] = $enabled;
                $user['updated_at'] = date('c');
                break;
            }
        }
        unset($user);

        $this->db->writeUsers($users);

        return true;
    }

    /**
     * Get all users
     *
     * @return array
     * @throws \Exception
     */
    public function getAll(): array
    {
        $users = $this->db->readUsers();
        return $users['users'];
    }

    /**
     * Get users by group ID
     *
     * @param string $groupId Group ID
     * @return array
     * @throws \Exception
     */
    public function getByGroupId(string $groupId): array
    {
        $users = $this->db->readUsers();
        $groupUsers = [];

        foreach ($users['users'] as $user) {
            if (in_array($groupId, $user['group_ids'])) {
                $groupUsers[] = $user;
            }
        }

        return $groupUsers;
    }

    /**
     * Generate master key hash from password
     *
     * @param string $password Password
     * @return string Master key hash
     * @throws \Exception
     */
    private function generateMasterKeyHash(string $password): string
    {
        $salt = Encryption::generateSalt();
        return Encryption::deriveKey($password, $salt);
    }
}
