<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Security\Encryption;
use PassGram\Helpers\Validator;

/**
 * Share Model
 *
 * Handles credential sharing between users.
 * Shared credential data is encrypted with the recipient's AES-256-GCM key
 * (the same key used to protect their own credentials) so no PGP key pair
 * is required on either side.
 */
class Share
{
    private Database $db;
    private Validator $validator;

    public function __construct(Database $db, Validator $validator)
    {
        $this->db = $db;
        $this->validator = $validator;
    }

    /**
     * Share a credential with another user.
     *
     * @param string $credentialId   Credential ID
     * @param string $ownerId        Owner user ID
     * @param string $targetUserId   Target user ID
     * @param array  $credentialData Decrypted credential data to share
     * @param string $encryptionKey  Recipient's 32-byte AES encryption key
     * @param array  $permissions    Optional permission overrides
     * @return array Created share record
     * @throws \Exception
     */
    public function create(
        string $credentialId,
        string $ownerId,
        string $targetUserId,
        array  $credentialData,
        string $encryptionKey,
        array  $permissions = []
    ): array {
        $encryption    = new Encryption($encryptionKey);
        $encryptedData = $encryption->encryptData($credentialData);

        $share = [
            'id'                       => Validator::generateUUID(),
            'credential_id'            => $credentialId,
            'owner_id'                 => $ownerId,
            'shared_with_user_id'      => $targetUserId,
            'shared_at'                => date('c'),
            'permissions'              => array_merge([
                'can_view'    => true,
                'can_edit'    => false,
                'can_reshare' => false,
            ], $permissions),
            'encrypted_credential_data' => $encryptedData,
            'revoked'                  => false,
            'revoked_at'               => null,
        ];

        $this->db->writeShare($share['id'], $share);

        return $share;
    }

    /**
     * Retrieve and decrypt a shared credential for its recipient.
     *
     * @param string $shareId       Share ID
     * @param string $userId        Requesting user ID (must match shared_with_user_id)
     * @param string $encryptionKey Recipient's 32-byte AES encryption key
     * @return array|null ['share' => array, 'credential' => array]
     * @throws \Exception
     */
    public function get(string $shareId, string $userId, string $encryptionKey): ?array
    {
        $share = $this->db->readShare($shareId);

        if (!$share) {
            return null;
        }

        if ($share['shared_with_user_id'] !== $userId) {
            throw new \Exception('Access denied');
        }

        if ($share['revoked']) {
            throw new \Exception('This share has been revoked');
        }

        $encryption     = new Encryption($encryptionKey);
        $credentialData = $encryption->decryptData($share['encrypted_credential_data']);

        return [
            'share'      => $share,
            'credential' => $credentialData,
        ];
    }

    /**
     * Revoke a share (owner only).
     *
     * @param string $shareId Share ID
     * @param string $ownerId Owner user ID
     * @return bool
     * @throws \Exception
     */
    public function revoke(string $shareId, string $ownerId): bool
    {
        $share = $this->db->readShare($shareId);

        if (!$share) {
            throw new \Exception('Share not found');
        }

        if ($share['owner_id'] !== $ownerId) {
            throw new \Exception('Only the owner can revoke this share');
        }

        $share['revoked']    = true;
        $share['revoked_at'] = date('c');

        $this->db->writeShare($shareId, $share);

        return true;
    }

    /**
     * Get all shares for a specific credential (owned by $ownerId).
     *
     * @param string $credentialId Credential ID
     * @param string $ownerId      Owner user ID
     * @return array
     * @throws \Exception
     */
    public function getByCredentialId(string $credentialId, string $ownerId): array
    {
        $shareIds = $this->db->listShares();
        $shares   = [];

        foreach ($shareIds as $shareId) {
            $share = $this->db->readShare($shareId);

            if ($share &&
                $share['credential_id'] === $credentialId &&
                $share['owner_id']      === $ownerId) {
                $shares[] = $share;
            }
        }

        return $shares;
    }

    /**
     * Get all active shares targeting a user.
     *
     * @param string $userId Recipient user ID
     * @return array
     * @throws \Exception
     */
    public function getSharedWithUser(string $userId): array
    {
        $shareIds = $this->db->listShares();
        $shares   = [];

        foreach ($shareIds as $shareId) {
            $share = $this->db->readShare($shareId);

            if ($share &&
                $share['shared_with_user_id'] === $userId &&
                !$share['revoked']) {
                $shares[] = $share;
            }
        }

        return $shares;
    }

    /**
     * Get all shares created by a user.
     *
     * @param string $userId Owner user ID
     * @return array
     * @throws \Exception
     */
    public function getByOwnerId(string $userId): array
    {
        $shareIds = $this->db->listShares();
        $shares   = [];

        foreach ($shareIds as $shareId) {
            $share = $this->db->readShare($shareId);

            if ($share && $share['owner_id'] === $userId) {
                $shares[] = $share;
            }
        }

        return $shares;
    }

    /**
     * Check whether a user has access to a share.
     *
     * @param string $shareId Share ID
     * @param string $userId  User ID
     * @return bool
     * @throws \Exception
     */
    public function hasAccess(string $shareId, string $userId): bool
    {
        $share = $this->db->readShare($shareId);

        if (!$share || $share['revoked']) {
            return false;
        }

        return $share['shared_with_user_id'] === $userId;
    }

    /**
     * Update permissions on a share (owner only).
     *
     * @param string $shareId     Share ID
     * @param string $ownerId     Owner user ID
     * @param array  $permissions New permissions
     * @return array Updated share
     * @throws \Exception
     */
    public function updatePermissions(string $shareId, string $ownerId, array $permissions): array
    {
        $share = $this->db->readShare($shareId);

        if (!$share) {
            throw new \Exception('Share not found');
        }

        if ($share['owner_id'] !== $ownerId) {
            throw new \Exception('Only the owner can update permissions');
        }

        $share['permissions'] = array_merge($share['permissions'], $permissions);

        $this->db->writeShare($shareId, $share);

        return $share;
    }

    /**
     * Permanently delete a share record (owner only).
     *
     * @param string $shareId Share ID
     * @param string $ownerId Owner user ID
     * @return bool
     * @throws \Exception
     */
    public function delete(string $shareId, string $ownerId): bool
    {
        $share = $this->db->readShare($shareId);

        if (!$share) {
            return true;
        }

        if ($share['owner_id'] !== $ownerId) {
            throw new \Exception('Only the owner can delete this share');
        }

        return $this->db->deleteShare($shareId);
    }
}
