<?php

namespace PassGram\Models;

use PassGram\Core\Database;

/**
 * PublicShare Model
 *
 * Handles one-time public credential share links.
 * Each link is identified by a 64-char cryptographically random hex token.
 * The link self-destructs on first access: once consumed it cannot be viewed again.
 */
class PublicShare
{
    private Database $db;

    public function __construct(Database $db)
    {
        $this->db = $db;
    }

    /**
     * Create a new public share link.
     *
     * @param string $ownerId       User who owns the credential
     * @param string $credentialId  Credential ID
     * @param array  $credentialData Decrypted credential fields to share
     * @return array Created share record (includes 'token')
     * @throws \Exception
     */
    public function create(string $ownerId, string $credentialId, array $credentialData): array
    {
        $token = bin2hex(random_bytes(32)); // 256-bit, URL-safe hex

        $share = [
            'token'           => $token,
            'credential_id'   => $credentialId,
            'owner_id'        => $ownerId,
            'credential_data' => $credentialData,
            'created_at'      => date('c'),
            'viewed'          => false,
            'viewed_at'       => null,
        ];

        $this->db->writePublicShare($token, $share);

        return $share;
    }

    /**
     * Consume a public share on first view.
     * Marks the link as viewed (burned). Returns null if already consumed or not found.
     *
     * @param string $token Share token
     * @return array|null Share data, or null if already used / not found
     * @throws \Exception
     */
    public function consume(string $token): ?array
    {
        $share = $this->db->readPublicShare($token);

        if (!$share) {
            return null;
        }

        if ($share['viewed']) {
            return null; // Already burned
        }

        $share['viewed']    = true;
        $share['viewed_at'] = date('c');
        $this->db->writePublicShare($token, $share);

        return $share;
    }

    /**
     * Get all public shares created by a user.
     *
     * @param string $ownerId Owner user ID
     * @return array
     * @throws \Exception
     */
    public function getByOwnerId(string $ownerId): array
    {
        $tokens = $this->db->listPublicShares();
        $shares = [];

        foreach ($tokens as $token) {
            $share = $this->db->readPublicShare($token);
            if ($share && $share['owner_id'] === $ownerId) {
                $shares[] = $share;
            }
        }

        // Newest first
        usort($shares, fn($a, $b) => strcmp($b['created_at'], $a['created_at']));

        return $shares;
    }

    /**
     * Delete a public share (owner only).
     *
     * @param string $token   Share token
     * @param string $ownerId Owner user ID
     * @return bool
     * @throws \Exception
     */
    public function delete(string $token, string $ownerId): bool
    {
        $share = $this->db->readPublicShare($token);

        if (!$share) {
            return true; // Already gone
        }

        if ($share['owner_id'] !== $ownerId) {
            throw new \Exception('Only the owner can delete this public share');
        }

        return $this->db->deletePublicShare($token);
    }
}
