<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Security\PGP;
use PassGram\Helpers\Validator;

/**
 * PGPKey Model
 *
 * Handles storage and group sharing of public PGP keys.
 * Public keys are stored unencrypted (they are public by nature)
 * but the storage file itself is encrypted with the master application key.
 */
class PGPKey
{
    private Database $db;
    private Validator $validator;

    public function __construct(Database $db, Validator $validator)
    {
        $this->db = $db;
        $this->validator = $validator;
    }

    /**
     * Store a new public PGP key
     *
     * @param string $userId User ID (owner)
     * @param array $data Key data (label, public_key, owner_name, owner_email, notes)
     * @return array Created key record
     * @throws \Exception
     */
    public function create(string $userId, array $data): array
    {
        if (!$this->validator->required($data['label'] ?? '', 'label')) {
            throw new \Exception($this->validator->getFirstError());
        }

        if (empty($data['public_key'])) {
            throw new \Exception('Public key is required');
        }

        // Validate the public key format
        if (!PGP::validatePublicKey($data['public_key'])) {
            throw new \Exception('Invalid public key format. Accepted formats: PEM/X.509, OpenPGP (.asc), SSH public key, PuTTY (.ppk), OpenSSH.');
        }

        // Extract key info
        $keyInfo = PGP::getKeyInfo($data['public_key']);
        $fingerprint = $keyInfo['fingerprint'];

        $key = [
            'id' => Validator::generateUUID(),
            'label' => $data['label'],
            'owner_name' => $data['owner_name'] ?? '',
            'owner_email' => $data['owner_email'] ?? '',
            'public_key' => $data['public_key'],
            'fingerprint' => $fingerprint,
            'algorithm' => $keyInfo['type'],
            'key_bits' => $keyInfo['bits'],
            'notes' => $data['notes'] ?? '',
            'tags' => $data['tags'] ?? [],
            'is_public' => $data['is_public'] ?? false,
            'is_shared' => false,
            'shared_with_groups' => [],
            'created_at' => date('c'),
            'updated_at' => date('c'),
        ];

        $keys = $this->db->readUserPGPKeys($userId);
        $keys['keys'][] = $key;
        $this->db->writeUserPGPKeys($userId, $keys);

        if ($key['is_public']) {
            $this->syncToCatalog($key, $userId);
        }

        return $key;
    }

    /**
     * Find a PGP key by ID in user's key store
     *
     * @param string $userId User ID
     * @param string $keyId Key ID
     * @return array|null
     * @throws \Exception
     */
    public function findById(string $userId, string $keyId): ?array
    {
        $keys = $this->db->readUserPGPKeys($userId);

        foreach ($keys['keys'] as $key) {
            if ($key['id'] === $keyId) {
                return $key;
            }
        }

        return null;
    }

    /**
     * Get all PGP keys for user
     *
     * @param string $userId User ID
     * @return array
     * @throws \Exception
     */
    public function getAll(string $userId): array
    {
        $keys = $this->db->readUserPGPKeys($userId);
        return $keys['keys'];
    }

    /**
     * Update a PGP key
     *
     * @param string $userId User ID
     * @param string $keyId Key ID
     * @param array $data Data to update
     * @return array Updated key
     * @throws \Exception
     */
    public function update(string $userId, string $keyId, array $data): array
    {
        $keys = $this->db->readUserPGPKeys($userId);
        $updated = null;

        foreach ($keys['keys'] as &$key) {
            if ($key['id'] === $keyId) {
                if (isset($data['label'])) $key['label'] = $data['label'];
                if (isset($data['owner_name'])) $key['owner_name'] = $data['owner_name'];
                if (isset($data['owner_email'])) $key['owner_email'] = $data['owner_email'];
                if (isset($data['notes'])) $key['notes'] = $data['notes'];
                if (isset($data['tags'])) $key['tags'] = $data['tags'];
                if (array_key_exists('is_public', $data)) $key['is_public'] = (bool) $data['is_public'];

                // If public key is being replaced, re-validate and re-extract info
                if (isset($data['public_key']) && !empty($data['public_key'])) {
                    if (!PGP::validatePublicKey($data['public_key'])) {
                        throw new \Exception('Invalid public key format. Accepted formats: PEM/X.509, OpenPGP (.asc), SSH public key, PuTTY (.ppk), OpenSSH.');
                    }
                    $keyInfo = PGP::getKeyInfo($data['public_key']);
                    $key['public_key'] = $data['public_key'];
                    $key['fingerprint'] = $keyInfo['fingerprint'];
                    $key['algorithm'] = $keyInfo['type'];
                    $key['key_bits'] = $keyInfo['bits'];
                }

                $key['updated_at'] = date('c');
                $updated = $key;
                break;
            }
        }

        if (!$updated) {
            throw new \Exception('PGP key not found');
        }

        $this->db->writeUserPGPKeys($userId, $keys);

        if (!empty($updated['is_public'])) {
            $this->syncToCatalog($updated, $userId);
        } else {
            $this->removeFromCatalog($keyId);
        }

        return $updated;
    }

    /**
     * Delete a PGP key
     *
     * @param string $userId User ID
     * @param string $keyId Key ID
     * @return bool
     * @throws \Exception
     */
    public function delete(string $userId, string $keyId): bool
    {
        $keys = $this->db->readUserPGPKeys($userId);
        $found = false;

        $keys['keys'] = array_filter($keys['keys'], function ($key) use ($keyId, &$found) {
            if ($key['id'] === $keyId) {
                $found = true;
                return false;
            }
            return true;
        });

        $keys['keys'] = array_values($keys['keys']);

        if (!$found) {
            throw new \Exception('PGP key not found');
        }

        $this->db->writeUserPGPKeys($userId, $keys);

        $this->removeFromCatalog($keyId);

        return true;
    }

    /**
     * Share PGP key with a group
     *
     * @param string $userId Owner user ID
     * @param string $keyId Key ID
     * @param string $groupId Group ID to share with
     * @return bool
     * @throws \Exception
     */
    public function shareWithGroup(string $userId, string $keyId, string $groupId): bool
    {
        $keys = $this->db->readUserPGPKeys($userId);
        $found = false;

        foreach ($keys['keys'] as &$key) {
            if ($key['id'] === $keyId) {
                if (!isset($key['shared_with_groups'])) {
                    $key['shared_with_groups'] = [];
                }

                // Check if already shared with this group
                $alreadyShared = false;
                foreach ($key['shared_with_groups'] as $share) {
                    if ($share['group_id'] === $groupId) {
                        $alreadyShared = true;
                        break;
                    }
                }

                if (!$alreadyShared) {
                    $key['shared_with_groups'][] = [
                        'group_id' => $groupId,
                        'shared_at' => date('c'),
                        'shared_by' => $userId,
                    ];
                }

                $key['is_shared'] = true;
                $key['updated_at'] = date('c');
                $found = true;
                break;
            }
        }

        if (!$found) {
            throw new \Exception('PGP key not found');
        }

        $this->db->writeUserPGPKeys($userId, $keys);

        return true;
    }

    /**
     * Revoke group access to PGP key
     *
     * @param string $userId Owner user ID
     * @param string $keyId Key ID
     * @param string $groupId Group ID to revoke
     * @return bool
     * @throws \Exception
     */
    public function revokeGroupShare(string $userId, string $keyId, string $groupId): bool
    {
        $keys = $this->db->readUserPGPKeys($userId);
        $found = false;

        foreach ($keys['keys'] as &$key) {
            if ($key['id'] === $keyId) {
                if (isset($key['shared_with_groups'])) {
                    $key['shared_with_groups'] = array_filter(
                        $key['shared_with_groups'],
                        function ($share) use ($groupId) {
                            return $share['group_id'] !== $groupId;
                        }
                    );
                    $key['shared_with_groups'] = array_values($key['shared_with_groups']);
                    $key['is_shared'] = !empty($key['shared_with_groups']);
                }

                $key['updated_at'] = date('c');
                $found = true;
                break;
            }
        }

        if (!$found) {
            throw new \Exception('PGP key not found');
        }

        $this->db->writeUserPGPKeys($userId, $keys);

        return true;
    }

    /**
     * Get all PGP keys accessible to user (owned + group shared)
     *
     * @param string $userId User ID
     * @param array $userGroupIds Array of group IDs user belongs to
     * @return array
     * @throws \Exception
     */
    public function getAllAccessible(string $userId, array $userGroupIds): array
    {
        $ownKeys = $this->getAll($userId);
        $groupSharedKeys = $this->getGroupSharedKeys($userId, $userGroupIds);

        foreach ($ownKeys as &$key) {
            $key['is_owner'] = true;
        }

        foreach ($groupSharedKeys as &$key) {
            $key['is_owner'] = false;
        }

        return array_merge($ownKeys, $groupSharedKeys);
    }

    /**
     * Get PGP keys shared with user's groups from other users
     *
     * @param string $userId Current user ID
     * @param array $userGroupIds Array of group IDs user belongs to
     * @return array
     * @throws \Exception
     */
    private function getGroupSharedKeys(string $userId, array $userGroupIds): array
    {
        $sharedKeys = [];
        $dataPath = $this->db->getDataPath();
        $pgpkeysPath = $dataPath . '/pgpkeys';

        if (!is_dir($pgpkeysPath)) {
            return [];
        }

        $files = glob($pgpkeysPath . '/*.json.enc');
        if ($files === false) {
            return [];
        }

        foreach ($files as $file) {
            $ownerId = basename($file, '.json.enc');

            if ($ownerId === $userId) {
                continue;
            }

            try {
                $keys = $this->db->readUserPGPKeys($ownerId);

                foreach ($keys['keys'] as $key) {
                    if (isset($key['shared_with_groups']) && !empty($key['shared_with_groups'])) {
                        foreach ($key['shared_with_groups'] as $groupShare) {
                            if (in_array($groupShare['group_id'], $userGroupIds)) {
                                $key['owner_id'] = $ownerId;
                                $key['shared_via_group'] = $groupShare['group_id'];
                                $sharedKeys[] = $key;
                                break;
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        return $sharedKeys;
    }

    /**
     * Find a PGP key by ID across all users (for group members viewing shared keys)
     *
     * @param string $keyId Key ID
     * @param array $userGroupIds Array of group IDs the requesting user belongs to
     * @return array|null ['key' => array, 'owner_id' => string] or null
     * @throws \Exception
     */
    public function findByIdAcrossUsers(string $keyId, array $userGroupIds): ?array
    {
        $dataPath = $this->db->getDataPath();
        $pgpkeysPath = $dataPath . '/pgpkeys';

        if (!is_dir($pgpkeysPath)) {
            return null;
        }

        $files = glob($pgpkeysPath . '/*.json.enc');
        if ($files === false) {
            return null;
        }

        foreach ($files as $file) {
            $ownerId = basename($file, '.json.enc');

            try {
                $keys = $this->db->readUserPGPKeys($ownerId);

                foreach ($keys['keys'] as $key) {
                    if ($key['id'] === $keyId) {
                        if (isset($key['shared_with_groups']) && !empty($key['shared_with_groups'])) {
                            foreach ($key['shared_with_groups'] as $groupShare) {
                                if (in_array($groupShare['group_id'], $userGroupIds)) {
                                    return [
                                        'key' => $key,
                                        'owner_id' => $ownerId,
                                        'shared_via_group' => $groupShare['group_id'],
                                    ];
                                }
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        return null;
    }

    /**
     * Build the safe public-facing record for the catalog.
     * Excludes: notes, tags, shared_with_groups, is_shared.
     */
    private function buildCatalogEntry(array $key, string $userId): array
    {
        return [
            'id'          => $key['id'],
            'label'       => $key['label'],
            'owner_name'  => $key['owner_name'] ?? '',
            'owner_email' => $key['owner_email'] ?? '',
            'public_key'  => $key['public_key'],
            'fingerprint' => $key['fingerprint'],
            'algorithm'   => $key['algorithm'],
            'key_bits'    => $key['key_bits'],
            'created_at'  => $key['created_at'],
            'updated_at'  => $key['updated_at'],
            'user_id'     => $userId,
        ];
    }

    /**
     * Add or update this key's entry in the public catalog.
     */
    private function syncToCatalog(array $key, string $userId): void
    {
        $catalog = $this->db->readPublicPGPKeyCatalog();
        $entry   = $this->buildCatalogEntry($key, $userId);
        $found   = false;

        foreach ($catalog['keys'] as &$existing) {
            if ($existing['id'] === $key['id']) {
                $existing = $entry;
                $found    = true;
                break;
            }
        }
        unset($existing);

        if (!$found) {
            $catalog['keys'][] = $entry;
        }

        $this->db->writePublicPGPKeyCatalog($catalog);
    }

    /**
     * Remove a key from the public catalog (on unmark or delete).
     */
    private function removeFromCatalog(string $keyId): void
    {
        $catalog = $this->db->readPublicPGPKeyCatalog();

        $catalog['keys'] = array_values(
            array_filter(
                $catalog['keys'],
                function ($entry) use ($keyId) {
                    return $entry['id'] !== $keyId;
                }
            )
        );

        $this->db->writePublicPGPKeyCatalog($catalog);
    }
}
