<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Security\Encryption;
use PassGram\Helpers\Validator;

/**
 * Note Model
 *
 * Handles notes on users (group context) and credentials.
 */
class Note
{
    private Database $db;
    private Validator $validator;

    public function __construct(Database $db, Validator $validator)
    {
        $this->db = $db;
        $this->validator = $validator;
    }

    /**
     * Create a note about a user (group context)
     *
     * @param string $authorId Author user ID
     * @param string $subjectUserId Subject user ID
     * @param string $groupId Group ID
     * @param string $content Note content
     * @param string $encryptionKey Encryption key
     * @return array Created note
     * @throws \Exception
     */
    public function createUserNote(
        string $authorId,
        string $subjectUserId,
        string $groupId,
        string $content,
        string $encryptionKey
    ): array {
        $note = [
            'id' => Validator::generateUUID(),
            'author_id' => $authorId,
            'subject_user_id' => $subjectUserId,
            'group_id' => $groupId,
            'content' => $this->encryptContent($content, $encryptionKey),
            'created_at' => date('c'),
            'updated_at' => date('c'),
        ];

        $notes = $this->db->readUserNotes();
        $notes['notes'][] = $note;
        $this->db->writeUserNotes($notes);

        return $note;
    }

    /**
     * Create a note on a credential
     *
     * @param string $credentialId Credential ID
     * @param string $userId User ID
     * @param string $content Note content
     * @param string $encryptionKey Encryption key
     * @return array Created note
     * @throws \Exception
     */
    public function createCredentialNote(
        string $credentialId,
        string $userId,
        string $content,
        string $encryptionKey
    ): array {
        $note = [
            'id' => Validator::generateUUID(),
            'credential_id' => $credentialId,
            'user_id' => $userId,
            'content' => $this->encryptContent($content, $encryptionKey),
            'created_at' => date('c'),
            'updated_at' => date('c'),
        ];

        $notes = $this->db->readCredentialNotes();
        $notes['notes'][] = $note;
        $this->db->writeCredentialNotes($notes);

        return $note;
    }

    /**
     * Get user notes for a specific user in a group
     *
     * @param string $subjectUserId Subject user ID
     * @param string $groupId Group ID
     * @param string $encryptionKey Encryption key
     * @param bool $decrypt Whether to decrypt content
     * @return array
     * @throws \Exception
     */
    public function getUserNotes(string $subjectUserId, string $groupId, string $encryptionKey, bool $decrypt = true): array
    {
        $notes = $this->db->readUserNotes();
        $results = [];

        foreach ($notes['notes'] as $note) {
            if ($note['subject_user_id'] === $subjectUserId && $note['group_id'] === $groupId) {
                if ($decrypt) {
                    $note['content'] = $this->decryptContent($note['content'], $encryptionKey);
                }
                $results[] = $note;
            }
        }

        return $results;
    }

    /**
     * Get notes for a credential
     *
     * @param string $credentialId Credential ID
     * @param string $userId User ID
     * @param string $encryptionKey Encryption key
     * @param bool $decrypt Whether to decrypt content
     * @return array
     * @throws \Exception
     */
    public function getCredentialNotes(string $credentialId, string $userId, string $encryptionKey, bool $decrypt = true): array
    {
        $notes = $this->db->readCredentialNotes();
        $results = [];

        foreach ($notes['notes'] as $note) {
            if ($note['credential_id'] === $credentialId && $note['user_id'] === $userId) {
                if ($decrypt) {
                    $note['content'] = $this->decryptContent($note['content'], $encryptionKey);
                }
                $results[] = $note;
            }
        }

        return $results;
    }

    /**
     * Get all notes by a group
     *
     * @param string $groupId Group ID
     * @param string $encryptionKey Encryption key
     * @param bool $decrypt Whether to decrypt content
     * @return array
     * @throws \Exception
     */
    public function getByGroupId(string $groupId, string $encryptionKey, bool $decrypt = true): array
    {
        $notes = $this->db->readUserNotes();
        $results = [];

        foreach ($notes['notes'] as $note) {
            if ($note['group_id'] === $groupId) {
                if ($decrypt) {
                    $note['content'] = $this->decryptContent($note['content'], $encryptionKey);
                }
                $results[] = $note;
            }
        }

        return $results;
    }

    /**
     * Update a user note
     *
     * @param string $noteId Note ID
     * @param string $authorId Author ID (for authorization)
     * @param string $content New content
     * @param string $encryptionKey Encryption key
     * @return array Updated note
     * @throws \Exception
     */
    public function updateUserNote(string $noteId, string $authorId, string $content, string $encryptionKey): array
    {
        $notes = $this->db->readUserNotes();
        $updated = null;

        foreach ($notes['notes'] as &$note) {
            if ($note['id'] === $noteId) {
                if ($note['author_id'] !== $authorId) {
                    throw new \Exception('Only the author can update this note');
                }

                $note['content'] = $this->encryptContent($content, $encryptionKey);
                $note['updated_at'] = date('c');
                $updated = $note;
                break;
            }
        }

        if (!$updated) {
            throw new \Exception('Note not found');
        }

        $this->db->writeUserNotes($notes);

        return $updated;
    }

    /**
     * Update a credential note
     *
     * @param string $noteId Note ID
     * @param string $userId User ID (for authorization)
     * @param string $content New content
     * @param string $encryptionKey Encryption key
     * @return array Updated note
     * @throws \Exception
     */
    public function updateCredentialNote(string $noteId, string $userId, string $content, string $encryptionKey): array
    {
        $notes = $this->db->readCredentialNotes();
        $updated = null;

        foreach ($notes['notes'] as &$note) {
            if ($note['id'] === $noteId) {
                if ($note['user_id'] !== $userId) {
                    throw new \Exception('Only the owner can update this note');
                }

                $note['content'] = $this->encryptContent($content, $encryptionKey);
                $note['updated_at'] = date('c');
                $updated = $note;
                break;
            }
        }

        if (!$updated) {
            throw new \Exception('Note not found');
        }

        $this->db->writeCredentialNotes($notes);

        return $updated;
    }

    /**
     * Delete a user note
     *
     * @param string $noteId Note ID
     * @param string $authorId Author ID (for authorization)
     * @return bool
     * @throws \Exception
     */
    public function deleteUserNote(string $noteId, string $authorId): bool
    {
        $notes = $this->db->readUserNotes();
        $found = false;

        $notes['notes'] = array_filter($notes['notes'], function ($note) use ($noteId, $authorId, &$found) {
            if ($note['id'] === $noteId) {
                if ($note['author_id'] !== $authorId) {
                    throw new \Exception('Only the author can delete this note');
                }
                $found = true;
                return false;
            }
            return true;
        });

        $notes['notes'] = array_values($notes['notes']);

        if (!$found) {
            throw new \Exception('Note not found');
        }

        $this->db->writeUserNotes($notes);

        return true;
    }

    /**
     * Delete a credential note
     *
     * @param string $noteId Note ID
     * @param string $userId User ID (for authorization)
     * @return bool
     * @throws \Exception
     */
    public function deleteCredentialNote(string $noteId, string $userId): bool
    {
        $notes = $this->db->readCredentialNotes();
        $found = false;

        $notes['notes'] = array_filter($notes['notes'], function ($note) use ($noteId, $userId, &$found) {
            if ($note['id'] === $noteId) {
                if ($note['user_id'] !== $userId) {
                    throw new \Exception('Only the owner can delete this note');
                }
                $found = true;
                return false;
            }
            return true;
        });

        $notes['notes'] = array_values($notes['notes']);

        if (!$found) {
            throw new \Exception('Note not found');
        }

        $this->db->writeCredentialNotes($notes);

        return true;
    }

    /**
     * Encrypt note content
     *
     * @param string $content Content to encrypt
     * @param string $key Encryption key
     * @return string Encrypted content
     * @throws \Exception
     */
    private function encryptContent(string $content, string $key): string
    {
        $encryption = new Encryption($key);
        return $encryption->encryptData($content);
    }

    /**
     * Decrypt note content
     *
     * @param string $encrypted Encrypted content
     * @param string $key Encryption key
     * @return string Decrypted content
     * @throws \Exception
     */
    private function decryptContent(string $encrypted, string $key): string
    {
        $encryption = new Encryption($key);
        return $encryption->decryptData($encrypted);
    }
}
