<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Core\Config;
use PassGram\Security\Encryption;
use PassGram\Helpers\Validator;

/**
 * Invite Model
 *
 * Handles invite code generation, validation, and tracking.
 */
class Invite
{
    private Database $db;
    private Config $config;
    private Validator $validator;

    public function __construct(Database $db, Config $config, Validator $validator)
    {
        $this->db = $db;
        $this->config = $config;
        $this->validator = $validator;
    }

    /**
     * Generate a new invite code
     *
     * @param string $groupId Group ID
     * @param string $createdBy User ID who created the invite
     * @param array $options Optional settings (expires_at, max_uses)
     * @return array Created invite
     * @throws \Exception
     */
    public function generate(string $groupId, string $createdBy, array $options = []): array
    {
        $invite = [
            'id' => Validator::generateUUID(),
            'code' => Encryption::generateToken(32), // 64 hex characters
            'group_id' => $groupId,
            'created_by' => $createdBy,
            'created_at' => date('c'),
            'expires_at' => $this->calculateExpiryDate($options['expires_at'] ?? null),
            'max_uses' => $options['max_uses'] ?? null,
            'used_count' => 0,
            'status' => 'active',
            'registrations' => [],
        ];

        $invites = $this->db->readInvites();
        $invites['invites'][] = $invite;
        $this->db->writeInvites($invites);

        return $invite;
    }

    /**
     * Validate an invite code
     *
     * @param string $code Invite code
     * @return array|null Invite data if valid, null otherwise
     * @throws \Exception
     */
    public function validate(string $code): ?array
    {
        if (!$this->validator->inviteCode($code)) {
            return null;
        }

        $invite = $this->findByCode($code);

        if (!$invite) {
            return null;
        }

        // Check status
        if ($invite['status'] !== 'active') {
            return null;
        }

        // Check expiry
        if ($invite['expires_at'] && strtotime($invite['expires_at']) < time()) {
            $this->markAsExpired($invite['id']);
            return null;
        }

        // Check max uses
        if ($invite['max_uses'] && $invite['used_count'] >= $invite['max_uses']) {
            return null;
        }

        return $invite;
    }

    /**
     * Use an invite code (mark as used)
     *
     * @param string $code Invite code
     * @param string $userId User ID who used the invite
     * @return bool
     * @throws \Exception
     */
    public function use(string $code, string $userId): bool
    {
        $invites = $this->db->readInvites();

        foreach ($invites['invites'] as &$invite) {
            if ($invite['code'] === $code) {
                $invite['used_count']++;
                $invite['registrations'][] = [
                    'user_id' => $userId,
                    'registered_at' => date('c'),
                ];

                // Check if max uses reached
                if ($invite['max_uses'] && $invite['used_count'] >= $invite['max_uses']) {
                    $invite['status'] = 'expired';
                }

                break;
            }
        }

        $this->db->writeInvites($invites);

        return true;
    }

    /**
     * Revoke an invite
     *
     * @param string $id Invite ID
     * @return bool
     * @throws \Exception
     */
    public function revoke(string $id): bool
    {
        $invites = $this->db->readInvites();

        foreach ($invites['invites'] as &$invite) {
            if ($invite['id'] === $id) {
                $invite['status'] = 'revoked';
                break;
            }
        }

        $this->db->writeInvites($invites);

        return true;
    }

    /**
     * Find invite by code
     *
     * @param string $code Invite code
     * @return array|null
     * @throws \Exception
     */
    public function findByCode(string $code): ?array
    {
        $invites = $this->db->readInvites();

        foreach ($invites['invites'] as $invite) {
            if ($invite['code'] === $code) {
                return $invite;
            }
        }

        return null;
    }

    /**
     * Find invite by ID
     *
     * @param string $id Invite ID
     * @return array|null
     * @throws \Exception
     */
    public function findById(string $id): ?array
    {
        $invites = $this->db->readInvites();

        foreach ($invites['invites'] as $invite) {
            if ($invite['id'] === $id) {
                return $invite;
            }
        }

        return null;
    }

    /**
     * Get invites for a group
     *
     * @param string $groupId Group ID
     * @return array
     * @throws \Exception
     */
    public function getByGroupId(string $groupId): array
    {
        $invites = $this->db->readInvites();
        $groupInvites = [];

        foreach ($invites['invites'] as $invite) {
            if ($invite['group_id'] === $groupId) {
                $groupInvites[] = $invite;
            }
        }

        return $groupInvites;
    }

    /**
     * Get invites created by user
     *
     * @param string $userId User ID
     * @return array
     * @throws \Exception
     */
    public function getByCreatorId(string $userId): array
    {
        $invites = $this->db->readInvites();
        $userInvites = [];

        foreach ($invites['invites'] as $invite) {
            if ($invite['created_by'] === $userId) {
                $userInvites[] = $invite;
            }
        }

        return $userInvites;
    }

    /**
     * Clean up expired invites
     *
     * @return int Number of invites cleaned up
     * @throws \Exception
     */
    public function cleanupExpired(): int
    {
        $invites = $this->db->readInvites();
        $count = 0;

        foreach ($invites['invites'] as &$invite) {
            if ($invite['status'] === 'active' &&
                $invite['expires_at'] &&
                strtotime($invite['expires_at']) < time()) {
                $invite['status'] = 'expired';
                $count++;
            }
        }

        if ($count > 0) {
            $this->db->writeInvites($invites);
        }

        return $count;
    }

    /**
     * Mark invite as expired
     *
     * @param string $id Invite ID
     * @return void
     * @throws \Exception
     */
    private function markAsExpired(string $id): void
    {
        $invites = $this->db->readInvites();

        foreach ($invites['invites'] as &$invite) {
            if ($invite['id'] === $id) {
                $invite['status'] = 'expired';
                break;
            }
        }

        $this->db->writeInvites($invites);
    }

    /**
     * Calculate expiry date
     *
     * @param string|null $expiresAt Custom expiry date or null for default
     * @return string|null ISO 8601 date or null
     */
    private function calculateExpiryDate(?string $expiresAt): ?string
    {
        if ($expiresAt) {
            return $expiresAt;
        }

        $days = $this->config->get('security.invite_expiry_days', 7);

        if ($days > 0) {
            return date('c', strtotime("+$days days"));
        }

        return null;
    }
}
