<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Helpers\Validator;

/**
 * Group Model
 *
 * Handles group CRUD operations and group membership.
 */
class Group
{
    private Database $db;
    private Validator $validator;

    public function __construct(Database $db, Validator $validator)
    {
        $this->db = $db;
        $this->validator = $validator;
    }

    /**
     * Create a new group
     *
     * @param array $data Group data
     * @return array Created group
     * @throws \Exception
     */
    public function create(array $data): array
    {
        if (!$this->validator->required($data['name'] ?? '', 'name')) {
            throw new \Exception($this->validator->getFirstError());
        }

        if (!$this->validator->length($data['name'], 3, 100, 'name')) {
            throw new \Exception($this->validator->getFirstError());
        }

        $group = [
            'id' => Validator::generateUUID(),
            'name' => $data['name'],
            'description' => $data['description'] ?? '',
            'created_at' => date('c'),
            'created_by' => $data['created_by'],
            'owner_id' => $data['owner_id'] ?? $data['created_by'],
            'member_ids' => [$data['created_by']], // Creator is first member
            'settings' => [
                'allow_credential_sharing' => true,
                'require_pgp' => false,
            ],
        ];

        $groups = $this->db->readGroups();
        $groups['groups'][] = $group;
        $this->db->writeGroups($groups);

        return $group;
    }

    /**
     * Find group by ID
     *
     * @param string $id Group ID
     * @return array|null
     * @throws \Exception
     */
    public function findById(string $id): ?array
    {
        $groups = $this->db->readGroups();

        foreach ($groups['groups'] as $group) {
            if ($group['id'] === $id) {
                return $group;
            }
        }

        return null;
    }

    /**
     * Update group
     *
     * @param string $id Group ID
     * @param array $data Data to update
     * @return array Updated group
     * @throws \Exception
     */
    public function update(string $id, array $data): array
    {
        $groups = $this->db->readGroups();
        $updated = null;

        foreach ($groups['groups'] as &$group) {
            if ($group['id'] === $id) {
                if (isset($data['name'])) {
                    if (!$this->validator->length($data['name'], 3, 100, 'name')) {
                        throw new \Exception($this->validator->getFirstError());
                    }
                    $group['name'] = $data['name'];
                }

                if (isset($data['description'])) {
                    $group['description'] = $data['description'];
                }

                if (isset($data['settings'])) {
                    $group['settings'] = array_merge($group['settings'], $data['settings']);
                }

                $updated = $group;
                break;
            }
        }

        if (!$updated) {
            throw new \Exception('Group not found');
        }

        $this->db->writeGroups($groups);

        return $updated;
    }

    /**
     * Delete group
     *
     * @param string $id Group ID
     * @return bool
     * @throws \Exception
     */
    public function delete(string $id): bool
    {
        $groups = $this->db->readGroups();
        $found = false;

        $groups['groups'] = array_filter($groups['groups'], function ($group) use ($id, &$found) {
            if ($group['id'] === $id) {
                $found = true;
                return false;
            }
            return true;
        });

        $groups['groups'] = array_values($groups['groups']);

        if (!$found) {
            throw new \Exception('Group not found');
        }

        $this->db->writeGroups($groups);

        return true;
    }

    /**
     * Add member to group
     *
     * @param string $groupId Group ID
     * @param string $userId User ID
     * @return bool
     * @throws \Exception
     */
    public function addMember(string $groupId, string $userId): bool
    {
        $groups = $this->db->readGroups();

        foreach ($groups['groups'] as &$group) {
            if ($group['id'] === $groupId) {
                if (!in_array($userId, $group['member_ids'])) {
                    $group['member_ids'][] = $userId;
                }
                break;
            }
        }

        $this->db->writeGroups($groups);

        return true;
    }

    /**
     * Remove member from group
     *
     * @param string $groupId Group ID
     * @param string $userId User ID
     * @return bool
     * @throws \Exception
     */
    public function removeMember(string $groupId, string $userId): bool
    {
        $groups = $this->db->readGroups();

        foreach ($groups['groups'] as &$group) {
            if ($group['id'] === $groupId) {
                $group['member_ids'] = array_filter($group['member_ids'], function ($uid) use ($userId) {
                    return $uid !== $userId;
                });
                $group['member_ids'] = array_values($group['member_ids']);
                break;
            }
        }

        $this->db->writeGroups($groups);

        return true;
    }

    /**
     * Get all groups
     *
     * @return array
     * @throws \Exception
     */
    public function getAll(): array
    {
        $groups = $this->db->readGroups();
        return $groups['groups'];
    }

    /**
     * Get groups for user
     *
     * @param string $userId User ID
     * @return array
     * @throws \Exception
     */
    public function getByUserId(string $userId): array
    {
        $groups = $this->db->readGroups();
        $userGroups = [];

        foreach ($groups['groups'] as $group) {
            if (in_array($userId, $group['member_ids'])) {
                $userGroups[] = $group;
            }
        }

        return $userGroups;
    }

    /**
     * Check if user is member of group
     *
     * @param string $groupId Group ID
     * @param string $userId User ID
     * @return bool
     * @throws \Exception
     */
    public function isMember(string $groupId, string $userId): bool
    {
        $group = $this->findById($groupId);

        if (!$group) {
            return false;
        }

        return in_array($userId, $group['member_ids']);
    }

    /**
     * Check if user is owner of group
     *
     * @param string $groupId Group ID
     * @param string $userId User ID
     * @return bool
     * @throws \Exception
     */
    public function isOwner(string $groupId, string $userId): bool
    {
        $group = $this->findById($groupId);

        if (!$group) {
            return false;
        }

        return $group['owner_id'] === $userId;
    }

    /**
     * Transfer ownership of group
     *
     * @param string $groupId Group ID
     * @param string $newOwnerId New owner user ID
     * @return bool
     * @throws \Exception
     */
    public function transferOwnership(string $groupId, string $newOwnerId): bool
    {
        $groups = $this->db->readGroups();
        $found = false;

        foreach ($groups['groups'] as &$group) {
            if ($group['id'] === $groupId) {
                // Verify new owner is a member
                if (!in_array($newOwnerId, $group['member_ids'])) {
                    throw new \Exception('New owner must be a member of the group');
                }

                $group['owner_id'] = $newOwnerId;
                $found = true;
                break;
            }
        }

        if (!$found) {
            throw new \Exception('Group not found');
        }

        $this->db->writeGroups($groups);

        return true;
    }
}
