<?php

namespace PassGram\Models;

use PassGram\Core\Database;
use PassGram\Security\Encryption;
use PassGram\Helpers\Validator;

/**
 * Credential Model
 *
 * Handles credential storage with field-level encryption.
 */
class Credential
{
    private Database $db;
    private Validator $validator;

    public function __construct(Database $db, Validator $validator)
    {
        $this->db = $db;
        $this->validator = $validator;
    }

    /**
     * Create a new credential
     *
     * @param string $userId User ID
     * @param array $data Credential data
     * @param string $encryptionKey User's encryption key
     * @return array Created credential
     * @throws \Exception
     */
    public function create(string $userId, array $data, string $encryptionKey): array
    {
        if (!$this->validator->required($data['title'] ?? '', 'title')) {
            throw new \Exception($this->validator->getFirstError());
        }

        $credential = [
            'id' => Validator::generateUUID(),
            'title' => $data['title'],
            'type' => $data['type'] ?? 'password',
            'username' => $data['username'] ?? null,
            'password' => null,
            'url' => $data['url'] ?? null,
            'notes' => $data['notes'] ?? '',
            'custom_fields' => $data['custom_fields'] ?? [],
            'tags' => $data['tags'] ?? [],
            'folder' => $data['folder'] ?? null,
            'is_shared' => false,
            'shared_with' => [],
            'shared_with_groups' => [],
            'created_at' => date('c'),
            'updated_at' => date('c'),
            'accessed_at' => date('c'),
        ];

        // Encrypt sensitive fields
        if (isset($data['password']) && $data['password'] !== '') {
            $credential['password'] = $this->encryptField($data['password'], $encryptionKey);
        }

        // Encrypt custom fields
        if (!empty($credential['custom_fields'])) {
            foreach ($credential['custom_fields'] as &$field) {
                if ($field['type'] === 'password') {
                    $field['value'] = $this->encryptField($field['value'], $encryptionKey);
                }
            }
        }

        // Add to user's credentials
        $credentials = $this->db->readUserCredentials($userId);
        $credentials['credentials'][] = $credential;
        $this->db->writeUserCredentials($userId, $credentials);

        return $credential;
    }

    /**
     * Find credential by ID
     *
     * @param string $userId User ID
     * @param string $credentialId Credential ID
     * @param string $encryptionKey User's encryption key
     * @param bool $decrypt Whether to decrypt sensitive fields
     * @return array|null
     * @throws \Exception
     */
    public function findById(string $userId, string $credentialId, string $encryptionKey, bool $decrypt = false): ?array
    {
        $credentials = $this->db->readUserCredentials($userId);

        foreach ($credentials['credentials'] as $credential) {
            if ($credential['id'] === $credentialId) {
                // Update last accessed
                $this->updateAccessTime($userId, $credentialId);

                if ($decrypt) {
                    return $this->decryptCredential($credential, $encryptionKey);
                }

                return $credential;
            }
        }

        return null;
    }

    /**
     * Get all credentials for user
     *
     * @param string $userId User ID
     * @param bool $decrypt Whether to decrypt sensitive fields
     * @param string|null $encryptionKey User's encryption key (required if decrypt=true)
     * @return array
     * @throws \Exception
     */
    public function getAll(string $userId, bool $decrypt = false, ?string $encryptionKey = null): array
    {
        $credentials = $this->db->readUserCredentials($userId);

        if ($decrypt && $encryptionKey) {
            $decrypted = [];
            foreach ($credentials['credentials'] as $credential) {
                $decrypted[] = $this->decryptCredential($credential, $encryptionKey);
            }
            return $decrypted;
        }

        return $credentials['credentials'];
    }

    /**
     * Update credential
     *
     * @param string $userId User ID
     * @param string $credentialId Credential ID
     * @param array $data Data to update
     * @param string $encryptionKey User's encryption key
     * @return array Updated credential
     * @throws \Exception
     */
    public function update(string $userId, string $credentialId, array $data, string $encryptionKey): array
    {
        $credentials = $this->db->readUserCredentials($userId);
        $updated = null;

        foreach ($credentials['credentials'] as &$credential) {
            if ($credential['id'] === $credentialId) {
                // Update allowed fields
                if (isset($data['title'])) $credential['title'] = $data['title'];
                if (isset($data['type'])) $credential['type'] = $data['type'];
                if (isset($data['username'])) $credential['username'] = $data['username'];
                if (isset($data['url'])) $credential['url'] = $data['url'];
                if (isset($data['notes'])) $credential['notes'] = $data['notes'];
                if (isset($data['tags'])) $credential['tags'] = $data['tags'];
                if (isset($data['folder'])) $credential['folder'] = $data['folder'];

                // Update password if provided
                if (isset($data['password']) && $data['password'] !== '') {
                    $credential['password'] = $this->encryptField($data['password'], $encryptionKey);
                }

                // Update custom fields
                if (isset($data['custom_fields'])) {
                    $credential['custom_fields'] = $data['custom_fields'];
                    foreach ($credential['custom_fields'] as &$field) {
                        if ($field['type'] === 'password') {
                            $field['value'] = $this->encryptField($field['value'], $encryptionKey);
                        }
                    }
                }

                $credential['updated_at'] = date('c');
                $updated = $credential;
                break;
            }
        }

        if (!$updated) {
            throw new \Exception('Credential not found');
        }

        $this->db->writeUserCredentials($userId, $credentials);

        return $updated;
    }

    /**
     * Delete credential
     *
     * @param string $userId User ID
     * @param string $credentialId Credential ID
     * @return bool
     * @throws \Exception
     */
    public function delete(string $userId, string $credentialId): bool
    {
        $credentials = $this->db->readUserCredentials($userId);
        $found = false;

        $credentials['credentials'] = array_filter($credentials['credentials'], function ($credential) use ($credentialId, &$found) {
            if ($credential['id'] === $credentialId) {
                $found = true;
                return false;
            }
            return true;
        });

        $credentials['credentials'] = array_values($credentials['credentials']);

        if (!$found) {
            throw new \Exception('Credential not found');
        }

        $this->db->writeUserCredentials($userId, $credentials);

        return true;
    }

    /**
     * Search credentials
     *
     * @param string $userId User ID
     * @param string $query Search query
     * @return array Matching credentials
     * @throws \Exception
     */
    public function search(string $userId, string $query): array
    {
        $credentials = $this->db->readUserCredentials($userId);
        $results = [];

        $query = strtolower($query);

        foreach ($credentials['credentials'] as $credential) {
            if (
                stripos($credential['title'], $query) !== false ||
                stripos($credential['username'] ?? '', $query) !== false ||
                stripos($credential['url'] ?? '', $query) !== false ||
                stripos($credential['notes'], $query) !== false
            ) {
                $results[] = $credential;
            }
        }

        return $results;
    }

    /**
     * Get credentials by folder
     *
     * @param string $userId User ID
     * @param string $folder Folder name
     * @return array
     * @throws \Exception
     */
    public function getByFolder(string $userId, string $folder): array
    {
        $credentials = $this->db->readUserCredentials($userId);
        $results = [];

        foreach ($credentials['credentials'] as $credential) {
            if ($credential['folder'] === $folder) {
                $results[] = $credential;
            }
        }

        return $results;
    }

    /**
     * Get credentials by tag
     *
     * @param string $userId User ID
     * @param string $tag Tag name
     * @return array
     * @throws \Exception
     */
    public function getByTag(string $userId, string $tag): array
    {
        $credentials = $this->db->readUserCredentials($userId);
        $results = [];

        foreach ($credentials['credentials'] as $credential) {
            if (in_array($tag, $credential['tags'])) {
                $results[] = $credential;
            }
        }

        return $results;
    }

    /**
     * Update access time
     *
     * @param string $userId User ID
     * @param string $credentialId Credential ID
     * @return void
     * @throws \Exception
     */
    private function updateAccessTime(string $userId, string $credentialId): void
    {
        $credentials = $this->db->readUserCredentials($userId);

        foreach ($credentials['credentials'] as &$credential) {
            if ($credential['id'] === $credentialId) {
                $credential['accessed_at'] = date('c');
                break;
            }
        }

        $this->db->writeUserCredentials($userId, $credentials);
    }

    /**
     * Encrypt a field
     *
     * @param string $value Value to encrypt
     * @param string $key Encryption key
     * @return string Encrypted value
     * @throws \Exception
     */
    private function encryptField(string $value, string $key): string
    {
        $encryption = new Encryption($key);
        return $encryption->encryptData($value);
    }

    /**
     * Decrypt a field
     *
     * @param string $encrypted Encrypted value
     * @param string $key Encryption key
     * @return string Decrypted value
     * @throws \Exception
     */
    private function decryptField(string $encrypted, string $key): string
    {
        $encryption = new Encryption($key);
        return $encryption->decryptData($encrypted);
    }

    /**
     * Decrypt credential fields
     *
     * @param array $credential Credential data
     * @param string $key Encryption key
     * @return array Decrypted credential
     * @throws \Exception
     */
    private function decryptCredential(array $credential, string $key): array
    {
        if ($credential['password']) {
            $credential['password'] = $this->decryptField($credential['password'], $key);
        }

        if (!empty($credential['custom_fields'])) {
            foreach ($credential['custom_fields'] as &$field) {
                if ($field['type'] === 'password' && !empty($field['value'])) {
                    $field['value'] = $this->decryptField($field['value'], $key);
                }
            }
        }

        return $credential;
    }

    /**
     * Share credential with a group
     *
     * @param string $userId Owner user ID
     * @param string $credentialId Credential ID
     * @param string $groupId Group ID to share with
     * @param string $permission Permission level ('read' or 'write')
     * @return bool
     * @throws \Exception
     */
    public function shareWithGroup(string $userId, string $credentialId, string $groupId, string $permission = 'read'): bool
    {
        if (!in_array($permission, ['read', 'write'])) {
            throw new \Exception('Invalid permission level. Must be "read" or "write".');
        }

        $credentials = $this->db->readUserCredentials($userId);
        $found = false;

        foreach ($credentials['credentials'] as &$credential) {
            if ($credential['id'] === $credentialId) {
                // Initialize shared_with_groups if not exists
                if (!isset($credential['shared_with_groups'])) {
                    $credential['shared_with_groups'] = [];
                }

                // Check if already shared with this group
                $alreadyShared = false;
                foreach ($credential['shared_with_groups'] as &$share) {
                    if ($share['group_id'] === $groupId) {
                        // Update permission if already shared
                        $share['permission'] = $permission;
                        $share['updated_at'] = date('c');
                        $alreadyShared = true;
                        break;
                    }
                }

                // Add new group share if not already shared
                if (!$alreadyShared) {
                    $credential['shared_with_groups'][] = [
                        'group_id' => $groupId,
                        'permission' => $permission,
                        'shared_at' => date('c'),
                        'shared_by' => $userId,
                    ];
                }

                $credential['is_shared'] = true;
                $credential['updated_at'] = date('c');
                $found = true;
                break;
            }
        }

        if (!$found) {
            throw new \Exception('Credential not found');
        }

        $this->db->writeUserCredentials($userId, $credentials);

        return true;
    }

    /**
     * Revoke group access to credential
     *
     * @param string $userId Owner user ID
     * @param string $credentialId Credential ID
     * @param string $groupId Group ID to revoke access from
     * @return bool
     * @throws \Exception
     */
    public function revokeGroupShare(string $userId, string $credentialId, string $groupId): bool
    {
        $credentials = $this->db->readUserCredentials($userId);
        $found = false;

        foreach ($credentials['credentials'] as &$credential) {
            if ($credential['id'] === $credentialId) {
                if (isset($credential['shared_with_groups'])) {
                    $credential['shared_with_groups'] = array_filter(
                        $credential['shared_with_groups'],
                        function ($share) use ($groupId) {
                            return $share['group_id'] !== $groupId;
                        }
                    );
                    $credential['shared_with_groups'] = array_values($credential['shared_with_groups']);

                    // Update is_shared flag
                    $credential['is_shared'] = !empty($credential['shared_with_groups']) || !empty($credential['shared_with']);
                }

                $credential['updated_at'] = date('c');
                $found = true;
                break;
            }
        }

        if (!$found) {
            throw new \Exception('Credential not found');
        }

        $this->db->writeUserCredentials($userId, $credentials);

        return true;
    }

    /**
     * Get all credentials accessible to user (owned + group shared)
     *
     * @param string $userId User ID
     * @param array $userGroupIds Array of group IDs user belongs to
     * @param bool $decrypt Whether to decrypt sensitive fields
     * @param string|null $encryptionKey User's encryption key (required if decrypt=true)
     * @return array
     * @throws \Exception
     */
    public function getAllAccessible(string $userId, array $userGroupIds, bool $decrypt = false, ?string $encryptionKey = null): array
    {
        // Get user's own credentials
        $ownCredentials = $this->getAll($userId, $decrypt, $encryptionKey);

        // Get credentials shared with user's groups from all users
        $groupSharedCredentials = $this->getGroupSharedCredentials($userId, $userGroupIds, $decrypt, $encryptionKey);

        // Merge and mark ownership
        foreach ($ownCredentials as &$cred) {
            $cred['is_owner'] = true;
            $cred['access_level'] = 'write';
        }

        foreach ($groupSharedCredentials as &$cred) {
            $cred['is_owner'] = false;
            // access_level is already set in getGroupSharedCredentials
        }

        return array_merge($ownCredentials, $groupSharedCredentials);
    }

    /**
     * Get credentials shared with user's groups
     *
     * @param string $userId Current user ID
     * @param array $userGroupIds Array of group IDs user belongs to
     * @param bool $decrypt Whether to decrypt sensitive fields
     * @param string|null $encryptionKey User's encryption key (required if decrypt=true)
     * @return array
     * @throws \Exception
     */
    private function getGroupSharedCredentials(string $userId, array $userGroupIds, bool $decrypt = false, ?string $encryptionKey = null): array
    {
        $sharedCredentials = [];

        // Get all user credential files
        $dataPath = $this->db->getDataPath();
        $credentialsPath = $dataPath . '/credentials';

        if (!is_dir($credentialsPath)) {
            return [];
        }

        $files = glob($credentialsPath . '/*.json.enc');
        if ($files === false) {
            return [];
        }

        foreach ($files as $file) {
            $ownerId = basename($file, '.json.enc');

            // Skip user's own credentials (already included)
            if ($ownerId === $userId) {
                continue;
            }

            try {
                $credentials = $this->db->readUserCredentials($ownerId);

                foreach ($credentials['credentials'] as $credential) {
                    // Check if shared with any of user's groups
                    if (isset($credential['shared_with_groups']) && !empty($credential['shared_with_groups'])) {
                        foreach ($credential['shared_with_groups'] as $groupShare) {
                            if (in_array($groupShare['group_id'], $userGroupIds)) {
                                // Add metadata
                                $credential['owner_id'] = $ownerId;
                                $credential['access_level'] = $groupShare['permission'];
                                $credential['shared_via_group'] = $groupShare['group_id'];

                                if ($decrypt && $encryptionKey) {
                                    $credential = $this->decryptCredential($credential, $encryptionKey);
                                }

                                $sharedCredentials[] = $credential;
                                break; // Only add once even if shared with multiple of user's groups
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                // Skip files that can't be read
                continue;
            }
        }

        return $sharedCredentials;
    }

    /**
     * Find a credential by ID across all users' credential files
     *
     * Used when a group member needs to view a credential owned by another user.
     *
     * @param string $credentialId Credential ID
     * @param array $userGroupIds Array of group IDs the requesting user belongs to
     * @return array|null ['credential' => array, 'owner_id' => string] or null
     * @throws \Exception
     */
    public function findByIdAcrossUsers(string $credentialId, array $userGroupIds): ?array
    {
        $dataPath = $this->db->getDataPath();
        $credentialsPath = $dataPath . '/credentials';

        if (!is_dir($credentialsPath)) {
            return null;
        }

        $files = glob($credentialsPath . '/*.json.enc');
        if ($files === false) {
            return null;
        }

        foreach ($files as $file) {
            $ownerId = basename($file, '.json.enc');

            try {
                $credentials = $this->db->readUserCredentials($ownerId);

                foreach ($credentials['credentials'] as $credential) {
                    if ($credential['id'] === $credentialId) {
                        // Verify the requesting user has group access
                        if (isset($credential['shared_with_groups']) && !empty($credential['shared_with_groups'])) {
                            foreach ($credential['shared_with_groups'] as $groupShare) {
                                if (in_array($groupShare['group_id'], $userGroupIds)) {
                                    return [
                                        'credential' => $credential,
                                        'owner_id' => $ownerId,
                                        'access_level' => $groupShare['permission'],
                                        'shared_via_group' => $groupShare['group_id'],
                                    ];
                                }
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        return null;
    }

    /**
     * Check user's permission level for a credential
     *
     * @param string $credentialId Credential ID
     * @param string $ownerId Credential owner ID
     * @param string $userId User ID to check
     * @param array $userGroupIds Array of group IDs user belongs to
     * @return string|null 'read', 'write', or null if no access
     * @throws \Exception
     */
    public function getUserPermission(string $credentialId, string $ownerId, string $userId, array $userGroupIds): ?string
    {
        // Owner has write access
        if ($ownerId === $userId) {
            return 'write';
        }

        // Check group shares
        $credentials = $this->db->readUserCredentials($ownerId);

        foreach ($credentials['credentials'] as $credential) {
            if ($credential['id'] === $credentialId) {
                if (isset($credential['shared_with_groups']) && !empty($credential['shared_with_groups'])) {
                    foreach ($credential['shared_with_groups'] as $groupShare) {
                        if (in_array($groupShare['group_id'], $userGroupIds)) {
                            return $groupShare['permission'];
                        }
                    }
                }
                break;
            }
        }

        return null;
    }

    /**
     * Generate random password
     *
     * @param int $length Password length
     * @param bool $includeSymbols Include special characters
     * @return string Generated password
     * @throws \Exception
     */
    public static function generatePassword(int $length = 16, bool $includeSymbols = true): string
    {
        $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';

        if ($includeSymbols) {
            $chars .= '!@#$%^&*()_+-=[]{}|;:,.<>?';
        }

        $password = '';
        $charLength = strlen($chars);

        for ($i = 0; $i < $length; $i++) {
            $password .= $chars[random_int(0, $charLength - 1)];
        }

        return $password;
    }
}
