<?php

namespace PassGram\Helpers;

use PassGram\Core\Config;

/**
 * Validator Class
 *
 * Provides input validation methods for PassGram.
 */
class Validator
{
    private Config $config;
    private array $errors = [];

    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    /**
     * Validate username
     *
     * @param string $username Username to validate
     * @return bool
     */
    public function username(string $username): bool
    {
        // 3-32 characters, alphanumeric and underscore only
        if (!preg_match('/^[a-zA-Z0-9_]{3,32}$/', $username)) {
            $this->errors['username'] = 'Username must be 3-32 characters (letters, numbers, underscore only)';
            return false;
        }
        return true;
    }

    /**
     * Validate email
     *
     * @param string $email Email to validate
     * @return bool
     */
    public function email(string $email): bool
    {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->errors['email'] = 'Invalid email address';
            return false;
        }
        return true;
    }

    /**
     * Validate password
     *
     * @param string $password Password to validate
     * @return bool
     */
    public function password(string $password): bool
    {
        $minLength = $this->config->get('security.password_min_length', 1);

        if (strlen($password) < $minLength) {
            $this->errors['password'] = "Password must be at least $minLength characters";
            return false;
        }

        $requireUpper = $this->config->get('security.password_require_uppercase', false);
        $requireLower = $this->config->get('security.password_require_lowercase', false);
        $requireNumbers = $this->config->get('security.password_require_numbers', false);
        $requireSpecial = $this->config->get('security.password_require_special', false);

        if ($requireUpper && !preg_match('/[A-Z]/', $password)) {
            $this->errors['password'] = 'Password must contain at least one uppercase letter';
            return false;
        }

        if ($requireLower && !preg_match('/[a-z]/', $password)) {
            $this->errors['password'] = 'Password must contain at least one lowercase letter';
            return false;
        }

        if ($requireNumbers && !preg_match('/[0-9]/', $password)) {
            $this->errors['password'] = 'Password must contain at least one number';
            return false;
        }

        if ($requireSpecial && !preg_match('/[^a-zA-Z0-9]/', $password)) {
            $this->errors['password'] = 'Password must contain at least one special character';
            return false;
        }

        return true;
    }

    /**
     * Validate UUID v4
     *
     * @param string $uuid UUID to validate
     * @return bool
     */
    public function uuid(string $uuid): bool
    {
        $pattern = '/^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i';
        if (!preg_match($pattern, $uuid)) {
            $this->errors['uuid'] = 'Invalid UUID format';
            return false;
        }
        return true;
    }

    /**
     * Validate invite code
     *
     * @param string $code Invite code to validate
     * @return bool
     */
    public function inviteCode(string $code): bool
    {
        // 64 hex characters
        if (!preg_match('/^[0-9a-f]{64}$/i', $code)) {
            $this->errors['invite_code'] = 'Invalid invite code format';
            return false;
        }
        return true;
    }

    /**
     * Validate URL
     *
     * @param string $url URL to validate
     * @return bool
     */
    public function url(string $url): bool
    {
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            $this->errors['url'] = 'Invalid URL format';
            return false;
        }
        return true;
    }

    /**
     * Validate required field
     *
     * @param mixed $value Value to check
     * @param string $fieldName Field name for error message
     * @return bool
     */
    public function required($value, string $fieldName = 'field'): bool
    {
        if (empty($value) && $value !== '0') {
            $this->errors[$fieldName] = ucfirst($fieldName) . ' is required';
            return false;
        }
        return true;
    }

    /**
     * Validate string length
     *
     * @param string $value String to validate
     * @param int $min Minimum length
     * @param int $max Maximum length
     * @param string $fieldName Field name for error message
     * @return bool
     */
    public function length(string $value, int $min, int $max, string $fieldName = 'field'): bool
    {
        $length = strlen($value);
        if ($length < $min || $length > $max) {
            $this->errors[$fieldName] = ucfirst($fieldName) . " must be between $min and $max characters";
            return false;
        }
        return true;
    }

    /**
     * Validate that value is in array
     *
     * @param mixed $value Value to check
     * @param array $options Valid options
     * @param string $fieldName Field name for error message
     * @return bool
     */
    public function in($value, array $options, string $fieldName = 'field'): bool
    {
        if (!in_array($value, $options, true)) {
            $this->errors[$fieldName] = ucfirst($fieldName) . ' must be one of: ' . implode(', ', $options);
            return false;
        }
        return true;
    }

    /**
     * Get validation errors
     *
     * @return array
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * Get first error message
     *
     * @return string|null
     */
    public function getFirstError(): ?string
    {
        return empty($this->errors) ? null : reset($this->errors);
    }

    /**
     * Check if there are errors
     *
     * @return bool
     */
    public function hasErrors(): bool
    {
        return !empty($this->errors);
    }

    /**
     * Clear all errors
     *
     * @return void
     */
    public function clearErrors(): void
    {
        $this->errors = [];
    }

    /**
     * Generate UUID v4
     *
     * @return string
     */
    public static function generateUUID(): string
    {
        $data = random_bytes(16);
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant

        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }
}
