<?php

namespace PassGram\Helpers;

/**
 * Sanitizer Class
 *
 * Provides input sanitization methods for PassGram.
 * Prevents XSS and other injection attacks.
 */
class Sanitizer
{
    /**
     * Sanitize string for output (prevent XSS)
     *
     * @param string $value Value to sanitize
     * @return string Sanitized value
     */
    public static function string(string $value): string
    {
        return htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }

    /**
     * Sanitize email
     *
     * @param string $email Email to sanitize
     * @return string Sanitized email
     */
    public static function email(string $email): string
    {
        return filter_var($email, FILTER_SANITIZE_EMAIL) ?: '';
    }

    /**
     * Sanitize URL
     *
     * @param string $url URL to sanitize
     * @return string Sanitized URL
     */
    public static function url(string $url): string
    {
        return filter_var($url, FILTER_SANITIZE_URL) ?: '';
    }

    /**
     * Sanitize integer
     *
     * @param mixed $value Value to sanitize
     * @return int Sanitized integer
     */
    public static function int($value): int
    {
        return filter_var($value, FILTER_SANITIZE_NUMBER_INT) ?: 0;
    }

    /**
     * Sanitize float
     *
     * @param mixed $value Value to sanitize
     * @return float Sanitized float
     */
    public static function float($value): float
    {
        return filter_var($value, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION) ?: 0.0;
    }

    /**
     * Sanitize array of strings
     *
     * @param array $array Array to sanitize
     * @return array Sanitized array
     */
    public static function array(array $array): array
    {
        return array_map([self::class, 'string'], $array);
    }

    /**
     * Strip tags from string
     *
     * @param string $value Value to sanitize
     * @param string|null $allowedTags Allowed HTML tags
     * @return string Sanitized value
     */
    public static function stripTags(string $value, ?string $allowedTags = null): string
    {
        return strip_tags($value, $allowedTags);
    }

    /**
     * Sanitize filename
     *
     * @param string $filename Filename to sanitize
     * @return string Sanitized filename
     */
    public static function filename(string $filename): string
    {
        // Remove any path components
        $filename = basename($filename);

        // Remove special characters
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);

        // Limit length
        if (strlen($filename) > 255) {
            $filename = substr($filename, 0, 255);
        }

        return $filename;
    }

    /**
     * Sanitize path (prevent directory traversal)
     *
     * @param string $path Path to sanitize
     * @return string Sanitized path
     */
    public static function path(string $path): string
    {
        // Remove any ../ or ../
        $path = str_replace(['../', '..\\'], '', $path);

        // Remove any null bytes
        $path = str_replace("\0", '', $path);

        return $path;
    }

    /**
     * Trim whitespace
     *
     * @param string $value Value to trim
     * @return string Trimmed value
     */
    public static function trim(string $value): string
    {
        return trim($value);
    }

    /**
     * Sanitize for JSON output
     *
     * @param mixed $value Value to sanitize
     * @return mixed Sanitized value
     */
    public static function json($value)
    {
        if (is_string($value)) {
            return self::string($value);
        }

        if (is_array($value)) {
            return array_map([self::class, 'json'], $value);
        }

        return $value;
    }

    /**
     * Clean input (trim and sanitize)
     *
     * @param string $value Value to clean
     * @return string Cleaned value
     */
    public static function clean(string $value): string
    {
        return self::string(self::trim($value));
    }

    /**
     * Sanitize POST data
     *
     * @param array $data POST data
     * @return array Sanitized data
     */
    public static function post(array $data): array
    {
        $sanitized = [];
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $sanitized[$key] = self::array($value);
            } else {
                $sanitized[$key] = self::trim($value);
            }
        }
        return $sanitized;
    }

    /**
     * Remove null bytes from string
     *
     * @param string $value Value to clean
     * @return string Cleaned value
     */
    public static function removeNullBytes(string $value): string
    {
        return str_replace("\0", '', $value);
    }
}
