<?php

namespace PassGram\Helpers;

use PassGram\Core\Config;

/**
 * Logger Class
 *
 * Provides logging functionality for PassGram.
 * Logs security events, errors, and activities.
 */
class Logger
{
    private string $logFile;
    private string $errorLogFile;
    private bool $enabled;

    public function __construct(string $logFile, bool $enabled = true)
    {
        // Ensure we have an absolute path
        if (empty($logFile) || $logFile === '/') {
            throw new \Exception('Invalid log file path provided');
        }

        $this->logFile = $logFile;
        $this->errorLogFile = dirname($logFile) . '/error.log';
        $this->enabled = $enabled;

        // Ensure log directory exists
        $logDir = dirname($this->logFile);
        if (!empty($logDir) && $logDir !== '/' && !is_dir($logDir)) {
            @mkdir($logDir, 0700, true);
        }
    }

    /**
     * Log an info message
     *
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool
     */
    public function info(string $message, array $context = []): bool
    {
        return $this->log('INFO', $message, $context);
    }

    /**
     * Log a warning message
     *
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool
     */
    public function warning(string $message, array $context = []): bool
    {
        return $this->log('WARNING', $message, $context);
    }

    /**
     * Log an error message
     *
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool
     */
    public function error(string $message, array $context = []): bool
    {
        $result = $this->log('ERROR', $message, $context);

        // Also write to error log
        $this->writeToFile($this->errorLogFile, 'ERROR', $message, $context);

        return $result;
    }

    /**
     * Log a debug message
     *
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool
     */
    public function debug(string $message, array $context = []): bool
    {
        // Debug logging always enabled
        return $this->log('DEBUG', $message, $context);
    }

    /**
     * Log authentication attempt
     *
     * @param string $username Username
     * @param bool $success Success status
     * @param string $ip IP address
     * @return bool
     */
    public function authAttempt(string $username, bool $success, string $ip): bool
    {
        $status = $success ? 'SUCCESS' : 'FAILED';
        $message = "Authentication attempt for user '$username' from IP $ip: $status";

        return $this->log('AUTH', $message, [
            'username' => $username,
            'success' => $success,
            'ip' => $ip,
        ]);
    }

    /**
     * Log credential access
     *
     * @param string $userId User ID
     * @param string $credentialId Credential ID
     * @param string $action Action performed
     * @return bool
     */
    public function credentialAccess(string $userId, string $credentialId, string $action): bool
    {
        $message = "User $userId $action credential $credentialId";

        return $this->log('CREDENTIAL', $message, [
            'user_id' => $userId,
            'credential_id' => $credentialId,
            'action' => $action,
        ]);
    }

    /**
     * Log share operation
     *
     * @param string $ownerId Owner user ID
     * @param string $targetUserId Target user ID
     * @param string $credentialId Credential ID
     * @param string $action Action (share/revoke)
     * @return bool
     */
    public function shareOperation(string $ownerId, string $targetUserId, string $credentialId, string $action): bool
    {
        $message = "User $ownerId {$action}d credential $credentialId with user $targetUserId";

        return $this->log('SHARE', $message, [
            'owner_id' => $ownerId,
            'target_user_id' => $targetUserId,
            'credential_id' => $credentialId,
            'action' => $action,
        ]);
    }

    /**
     * Log admin action
     *
     * @param string $userId User ID
     * @param string $action Action performed
     * @param array $context Additional context
     * @return bool
     */
    public function adminAction(string $userId, string $action, array $context = []): bool
    {
        $message = "Admin user $userId performed action: $action";

        $context['user_id'] = $userId;
        $context['action'] = $action;

        return $this->log('ADMIN', $message, $context);
    }

    /**
     * Write log entry
     *
     * @param string $level Log level
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool
     */
    private function log(string $level, string $message, array $context = []): bool
    {
        if (!$this->enabled) {
            return false;
        }

        return $this->writeToFile($this->logFile, $level, $message, $context);
    }

    /**
     * Write to log file
     *
     * @param string $file Log file path
     * @param string $level Log level
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool
     */
    private function writeToFile(string $file, string $level, string $message, array $context): bool
    {
        $timestamp = date('Y-m-d H:i:s');
        $contextJson = !empty($context) ? ' ' . json_encode($context) : '';
        $entry = "[$timestamp] [$level] $message$contextJson" . PHP_EOL;

        // Check file size and rotate if necessary
        if (file_exists($file)) {
            $maxSize = 10485760; // 10MB default
            $filesize = @filesize($file);
            if ($filesize !== false && $filesize > $maxSize) {
                $this->rotateLog($file);
            }
        }

        // Append to log file
        $result = @file_put_contents($file, $entry, FILE_APPEND | LOCK_EX);

        return $result !== false;
    }

    /**
     * Rotate log file
     *
     * @param string $file Log file path
     * @return void
     */
    private function rotateLog(string $file): void
    {
        if (file_exists($file)) {
            $timestamp = date('Y-m-d_H-i-s');
            $rotatedFile = $file . '.' . $timestamp;
            rename($file, $rotatedFile);
        }
    }

    /**
     * Get recent log entries
     *
     * @param int $lines Number of lines to retrieve
     * @return array Log entries
     */
    public function getRecentEntries(int $lines = 100): array
    {
        if (!file_exists($this->logFile)) {
            return [];
        }

        $file = file($this->logFile);
        return array_slice($file, -$lines);
    }

    /**
     * Clear log file
     *
     * @return bool
     */
    public function clear(): bool
    {
        if (file_exists($this->logFile)) {
            return unlink($this->logFile);
        }
        return true;
    }
}
