<?php

namespace PassGram\Core;

use PassGram\Security\Encryption;
use PassGram\Security\PGPEncryption;

/**
 * Database Class
 *
 * Handles encrypted JSON file storage operations for PassGram.
 * Provides atomic writes, file locking, and transparent encryption/decryption.
 *
 * Supports two per-user encryption modes:
 *   - AES mode (default): files encrypted with the master application key
 *     using AES-256-GCM (*.json.enc).
 *   - PGP mode: files encrypted with the user's own RSA/EC key pair
 *     using hybrid RSA+AES-256-GCM (*.pgp.enc).
 *     Public-key stores are kept as plain JSON (*.json) since they are
 *     not sensitive by nature.
 */
class Database
{
    private Encryption $encryption;
    private array $config;
    private string $lockDir;

    /**
     * Optional PGP context used by readUserCredentials / writeUserCredentials
     * and the PGP-key equivalents when the user has PGP mode enabled.
     *
     * Shape: ['public_key' => string, 'encrypted_private_key' => string, 'passphrase' => string]
     */
    private ?array $pgpContext = null;

    /**
     * Constructor
     *
     * @param Encryption $encryption Encryption instance
     * @param array $config Database configuration
     */
    public function __construct(Encryption $encryption, array $config)
    {
        $this->encryption = $encryption;
        $this->config = $config;
        $this->lockDir = dirname($this->config['users']) . '/locks';

        // Create locks directory if it doesn't exist
        if (!is_dir($this->lockDir)) {
            mkdir($this->lockDir, 0700, true);
        }
    }

    /**
     * Read data from an encrypted JSON file
     *
     * @param string $filepath Path to the encrypted file
     * @param mixed $default Default value if file doesn't exist
     * @return mixed Decrypted and decoded data
     * @throws \Exception
     */
    public function read(string $filepath, $default = null)
    {
        // Return default if file doesn't exist
        if (!file_exists($filepath)) {
            return $default;
        }

        // Check if file is readable
        if (!is_readable($filepath)) {
            throw new \Exception("File is not readable: $filepath");
        }

        // Acquire shared lock for reading
        $fp = fopen($filepath, 'r');
        if ($fp === false) {
            throw new \Exception("Failed to open file for reading: $filepath");
        }

        try {
            if (!flock($fp, LOCK_SH)) {
                fclose($fp);
                throw new \Exception("Failed to acquire read lock: $filepath");
            }

            // Get file size
            $filesize = filesize($filepath);
            if ($filesize === false || $filesize === 0) {
                flock($fp, LOCK_UN);
                fclose($fp);
                return $default;
            }

            // Read encrypted content
            $encrypted = fread($fp, $filesize);
            if ($encrypted === false) {
                flock($fp, LOCK_UN);
                fclose($fp);
                throw new \Exception("Failed to read file: $filepath");
            }

            // Release lock and close file
            flock($fp, LOCK_UN);
            fclose($fp);

            // Decrypt and return data
            return $this->encryption->decryptData($encrypted);

        } catch (\Exception $e) {
            // Safely cleanup
            if (is_resource($fp)) {
                @flock($fp, LOCK_UN);
                @fclose($fp);
            }
            throw $e;
        }
    }

    /**
     * Write data to an encrypted JSON file (atomic operation)
     *
     * @param string $filepath Path to the encrypted file
     * @param mixed $data Data to encrypt and write
     * @return bool True on success
     * @throws \Exception
     */
    public function write(string $filepath, $data): bool
    {
        // Ensure directory exists
        $dir = dirname($filepath);
        if (!is_dir($dir)) {
            // Try strict mode first; fall back to 0755 for filesystems that
            // reject 0700 (e.g. pCloud Drive, some network mounts).
            if (!mkdir($dir, 0700, true) && !is_dir($dir)) {
                if (!mkdir($dir, 0755, true) && !is_dir($dir)) {
                    throw new \Exception("Failed to create directory: $dir");
                }
            }
        }

        // Encrypt data
        $encrypted = $this->encryption->encryptData($data);

        // Write to temporary file first (atomic write)
        $tempFile = $filepath . '.tmp.' . uniqid();

        $fp = fopen($tempFile, 'w');
        if ($fp === false) {
            throw new \Exception("Failed to open temporary file: $tempFile");
        }

        try {
            if (!flock($fp, LOCK_EX)) {
                throw new \Exception("Failed to acquire write lock: $tempFile");
            }

            $result = fwrite($fp, $encrypted);
            if ($result === false) {
                throw new \Exception("Failed to write to file: $tempFile");
            }

            fflush($fp);
            flock($fp, LOCK_UN);
            fclose($fp);

            // Set file permissions
            chmod($tempFile, 0600);

            // Atomic rename
            if (!rename($tempFile, $filepath)) {
                throw new \Exception("Failed to rename temporary file to: $filepath");
            }

            return true;

        } catch (\Exception $e) {
            flock($fp, LOCK_UN);
            fclose($fp);

            // Clean up temporary file
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }

            throw $e;
        }
    }

    /**
     * Update data in an encrypted JSON file
     * Reads, modifies with callback, then writes atomically
     *
     * @param string $filepath Path to the encrypted file
     * @param callable $callback Callback function to modify data
     * @param mixed $default Default value if file doesn't exist
     * @return mixed Modified data
     * @throws \Exception
     */
    public function update(string $filepath, callable $callback, $default = null)
    {
        // Read current data
        $data = $this->read($filepath, $default);

        // Apply callback
        $data = $callback($data);

        // Write back
        $this->write($filepath, $data);

        return $data;
    }

    /**
     * Delete an encrypted JSON file
     *
     * @param string $filepath Path to the file
     * @return bool True on success
     */
    public function delete(string $filepath): bool
    {
        if (file_exists($filepath)) {
            return unlink($filepath);
        }
        return true;
    }

    /**
     * Check if a file exists
     *
     * @param string $filepath Path to check
     * @return bool
     */
    public function exists(string $filepath): bool
    {
        return file_exists($filepath);
    }

    /**
     * Read users database
     *
     * @return array Users data
     * @throws \Exception
     */
    public function readUsers(): array
    {
        return $this->read(
            $this->config['users'],
            $this->config['defaults']['users']
        );
    }

    /**
     * Write users database
     *
     * @param array $data Users data
     * @return bool
     * @throws \Exception
     */
    public function writeUsers(array $data): bool
    {
        return $this->write($this->config['users'], $data);
    }

    /**
     * Read groups database
     *
     * @return array Groups data
     * @throws \Exception
     */
    public function readGroups(): array
    {
        return $this->read(
            $this->config['groups'],
            $this->config['defaults']['groups']
        );
    }

    /**
     * Write groups database
     *
     * @param array $data Groups data
     * @return bool
     * @throws \Exception
     */
    public function writeGroups(array $data): bool
    {
        return $this->write($this->config['groups'], $data);
    }

    /**
     * Read invites database
     *
     * @return array Invites data
     * @throws \Exception
     */
    public function readInvites(): array
    {
        return $this->read(
            $this->config['invites'],
            $this->config['defaults']['invites']
        );
    }

    /**
     * Write invites database
     *
     * @param array $data Invites data
     * @return bool
     * @throws \Exception
     */
    public function writeInvites(array $data): bool
    {
        return $this->write($this->config['invites'], $data);
    }

    // -------------------------------------------------------------------------
    // PGP context management
    // -------------------------------------------------------------------------

    /**
     * Set the PGP context used for per-user credential and key storage.
     * Call this before any readUserCredentials / writeUserCredentials (etc.)
     * when the authenticated user has PGP encryption mode enabled.
     *
     * The $ownerId parameter ensures PGP encryption is ONLY applied when
     * reading/writing that specific user's files. Access to other users'
     * files (e.g. for group-shared credentials) always falls back to AES.
     *
     * @param string $publicKey           PEM public key
     * @param string $encryptedPrivateKey Passphrase-protected private key blob
     * @param string $passphrase          PGP key passphrase (from session)
     * @param string $ownerId             User ID whose files use PGP mode
     * @return void
     */
    public function setPGPContext(string $publicKey, string $encryptedPrivateKey, string $passphrase, string $ownerId = ''): void
    {
        $this->pgpContext = [
            'public_key'            => $publicKey,
            'encrypted_private_key' => $encryptedPrivateKey,
            'passphrase'            => $passphrase,
            'owner_id'              => $ownerId,
        ];
    }

    /**
     * Return true only when the PGP context is set AND the given userId matches
     * the context owner (so other users' files are always read/written as AES).
     *
     * @param string $userId
     * @return bool
     */
    private function pgpContextAppliesTo(string $userId): bool
    {
        if ($this->pgpContext === null) {
            return false;
        }
        // If no owner_id was specified, apply to all (legacy / migration use)
        if ($this->pgpContext['owner_id'] === '') {
            return true;
        }
        return $this->pgpContext['owner_id'] === $userId;
    }

    /**
     * Clear the PGP context (revert to AES mode for subsequent calls).
     */
    public function clearPGPContext(): void
    {
        $this->pgpContext = null;
    }

    /**
     * Whether a PGP context is currently active.
     */
    public function hasPGPContext(): bool
    {
        return $this->pgpContext !== null;
    }

    // -------------------------------------------------------------------------
    // Per-user credential storage (AES or PGP depending on context)
    // -------------------------------------------------------------------------

    /**
     * Read user's credentials.
     *
     * When a PGP context is set the *.pgp.enc file is used; otherwise the
     * standard AES *.json.enc file is used.
     *
     * @param string $userId User ID
     * @return array Credentials data
     * @throws \Exception
     */
    public function readUserCredentials(string $userId): array
    {
        $default = $this->config['defaults']['credentials'];
        $default['user_id'] = $userId;

        if ($this->pgpContextAppliesTo($userId)) {
            return $this->readPGPFile(
                $this->config['credentials_dir'] . '/' . $userId . '.pgp.enc',
                $default
            );
        }

        $filepath = $this->config['credentials_dir'] . '/' . $userId . '.json.enc';
        return $this->read($filepath, $default);
    }

    /**
     * Write user's credentials.
     *
     * When a PGP context is set the *.pgp.enc file is written; otherwise the
     * standard AES *.json.enc file is written.
     *
     * @param string $userId User ID
     * @param array $data Credentials data
     * @return bool
     * @throws \Exception
     */
    public function writeUserCredentials(string $userId, array $data): bool
    {
        if ($this->pgpContextAppliesTo($userId)) {
            return $this->writePGPFile(
                $this->config['credentials_dir'] . '/' . $userId . '.pgp.enc',
                $data
            );
        }

        $filepath = $this->config['credentials_dir'] . '/' . $userId . '.json.enc';
        return $this->write($filepath, $data);
    }

    // -------------------------------------------------------------------------
    // Per-user PGP public-key storage (AES or plain JSON depending on context)
    // -------------------------------------------------------------------------

    /**
     * Read user's PGP public keys.
     *
     * When a PGP context is set the *.json file (plain JSON) is used because
     * public keys are not sensitive and need not be encrypted.
     * Otherwise the standard AES *.json.enc file is used.
     *
     * @param string $userId User ID
     * @return array PGP keys data
     * @throws \Exception
     */
    public function readUserPGPKeys(string $userId): array
    {
        $default = $this->config['defaults']['pgpkeys'];
        $default['user_id'] = $userId;

        if ($this->pgpContextAppliesTo($userId)) {
            return $this->readPlainJSON(
                $this->config['pgpkeys_dir'] . '/' . $userId . '.json',
                $default
            );
        }

        $filepath = $this->config['pgpkeys_dir'] . '/' . $userId . '.json.enc';
        return $this->read($filepath, $default);
    }

    /**
     * Write user's PGP public keys.
     *
     * When a PGP context is set the *.json file (plain JSON) is written.
     * Otherwise the standard AES *.json.enc file is written.
     *
     * @param string $userId User ID
     * @param array $data PGP keys data
     * @return bool
     * @throws \Exception
     */
    public function writeUserPGPKeys(string $userId, array $data): bool
    {
        if ($this->pgpContextAppliesTo($userId)) {
            return $this->writePlainJSON(
                $this->config['pgpkeys_dir'] . '/' . $userId . '.json',
                $data
            );
        }

        $filepath = $this->config['pgpkeys_dir'] . '/' . $userId . '.json.enc';
        return $this->write($filepath, $data);
    }

    // -------------------------------------------------------------------------
    // PGP-mode migration helpers (called by PGPController)
    // -------------------------------------------------------------------------

    /**
     * Migrate a user's credentials from AES to PGP encryption.
     * Reads the AES *.json.enc file and writes a PGP *.pgp.enc file.
     * The AES file is removed on success.
     *
     * Requires pgpContext to be set before calling.
     *
     * @param string $userId
     * @return bool True on success
     * @throws \Exception
     */
    public function migrateCredentialsToPGP(string $userId): bool
    {
        if ($this->pgpContext === null) {
            throw new \Exception('PGP context is not set');
        }

        $aesFile = $this->config['credentials_dir'] . '/' . $userId . '.json.enc';
        $pgpFile = $this->config['credentials_dir'] . '/' . $userId . '.pgp.enc';

        $default = $this->config['defaults']['credentials'];
        $default['user_id'] = $userId;

        // Read existing AES data directly (bypasses PGP context check)
        $data = $this->read($aesFile, $default);

        // Write PGP-encrypted version directly
        $this->writePGPFile($pgpFile, $data);

        // Remove the old AES file
        if (file_exists($aesFile)) {
            unlink($aesFile);
        }

        return true;
    }

    /**
     * Migrate a user's credentials from PGP back to AES encryption.
     * Reads the PGP *.pgp.enc file and writes a AES *.json.enc file.
     * The PGP file is removed on success.
     *
     * Requires pgpContext to be set before calling.
     *
     * @param string $userId
     * @return bool True on success
     * @throws \Exception
     */
    public function migrateCredentialsToAES(string $userId): bool
    {
        if ($this->pgpContext === null) {
            throw new \Exception('PGP context is not set');
        }

        $pgpFile = $this->config['credentials_dir'] . '/' . $userId . '.pgp.enc';
        $aesFile = $this->config['credentials_dir'] . '/' . $userId . '.json.enc';

        if (!file_exists($pgpFile)) {
            // Nothing to migrate
            return true;
        }

        $default = $this->config['defaults']['credentials'];
        $default['user_id'] = $userId;

        // Read PGP-encrypted data
        $data = $this->readPGPFile($pgpFile, $default);

        // Write AES-encrypted version (without PGP context)
        $this->write($aesFile, $data);

        // Remove the PGP file
        unlink($pgpFile);

        return true;
    }

    /**
     * Migrate a user's PGP public-key store from AES to plain JSON.
     * Reads the AES *.json.enc file and writes a plain *.json file.
     * The AES file is removed on success.
     *
     * @param string $userId
     * @return bool
     * @throws \Exception
     */
    public function migratePublicKeysToPGPMode(string $userId): bool
    {
        $aesFile = $this->config['pgpkeys_dir'] . '/' . $userId . '.json.enc';
        $jsonFile = $this->config['pgpkeys_dir'] . '/' . $userId . '.json';

        $default = $this->config['defaults']['pgpkeys'];
        $default['user_id'] = $userId;

        $data = $this->read($aesFile, $default);
        $this->writePlainJSON($jsonFile, $data);

        if (file_exists($aesFile)) {
            unlink($aesFile);
        }

        return true;
    }

    /**
     * Migrate a user's PGP public-key store from plain JSON back to AES.
     * Reads the *.json file and writes an AES *.json.enc file.
     * The plain JSON file is removed on success.
     *
     * @param string $userId
     * @return bool
     * @throws \Exception
     */
    public function migratePublicKeysToAES(string $userId): bool
    {
        $jsonFile = $this->config['pgpkeys_dir'] . '/' . $userId . '.json';
        $aesFile  = $this->config['pgpkeys_dir'] . '/' . $userId . '.json.enc';

        if (!file_exists($jsonFile)) {
            return true;
        }

        $default = $this->config['defaults']['pgpkeys'];
        $default['user_id'] = $userId;

        $data = $this->readPlainJSON($jsonFile, $default);
        $this->write($aesFile, $data);
        unlink($jsonFile);

        return true;
    }

    // -------------------------------------------------------------------------
    // Low-level PGP file I/O helpers
    // -------------------------------------------------------------------------

    /**
     * Read a PGP-encrypted file.
     *
     * @param string $filepath
     * @param mixed  $default
     * @return mixed
     * @throws \Exception
     */
    private function readPGPFile(string $filepath, $default = null)
    {
        if (!file_exists($filepath)) {
            return $default;
        }

        if (!is_readable($filepath)) {
            throw new \Exception("File is not readable: $filepath");
        }

        $fp = fopen($filepath, 'r');
        if ($fp === false) {
            throw new \Exception("Failed to open file for reading: $filepath");
        }

        try {
            if (!flock($fp, LOCK_SH)) {
                fclose($fp);
                throw new \Exception("Failed to acquire read lock: $filepath");
            }

            $filesize = filesize($filepath);
            if ($filesize === false || $filesize === 0) {
                flock($fp, LOCK_UN);
                fclose($fp);
                return $default;
            }

            $encrypted = fread($fp, $filesize);
            flock($fp, LOCK_UN);
            fclose($fp);

            if ($encrypted === false) {
                throw new \Exception("Failed to read file: $filepath");
            }

            return PGPEncryption::decryptData(
                $encrypted,
                $this->pgpContext['encrypted_private_key'],
                $this->pgpContext['passphrase']
            );

        } catch (\Exception $e) {
            if (is_resource($fp)) {
                @flock($fp, LOCK_UN);
                @fclose($fp);
            }
            throw $e;
        }
    }

    /**
     * Write a PGP-encrypted file (atomic write).
     *
     * @param string $filepath
     * @param mixed  $data
     * @return bool
     * @throws \Exception
     */
    private function writePGPFile(string $filepath, $data): bool
    {
        $dir = dirname($filepath);
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0700, true) && !is_dir($dir)) {
                if (!mkdir($dir, 0755, true) && !is_dir($dir)) {
                    throw new \Exception("Failed to create directory: $dir");
                }
            }
        }

        $encrypted = PGPEncryption::encryptData($data, $this->pgpContext['public_key']);

        $tempFile = $filepath . '.tmp.' . uniqid();
        $fp = fopen($tempFile, 'w');
        if ($fp === false) {
            throw new \Exception("Failed to open temporary file: $tempFile");
        }

        try {
            if (!flock($fp, LOCK_EX)) {
                throw new \Exception("Failed to acquire write lock: $tempFile");
            }

            if (fwrite($fp, $encrypted) === false) {
                throw new \Exception("Failed to write to file: $tempFile");
            }

            fflush($fp);
            flock($fp, LOCK_UN);
            fclose($fp);

            chmod($tempFile, 0600);

            if (!rename($tempFile, $filepath)) {
                throw new \Exception("Failed to rename temporary file to: $filepath");
            }

            return true;

        } catch (\Exception $e) {
            flock($fp, LOCK_UN);
            fclose($fp);
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }
            throw $e;
        }
    }

    /**
     * Read a plain (unencrypted) JSON file.
     *
     * @param string $filepath
     * @param mixed  $default
     * @return mixed
     * @throws \Exception
     */
    private function readPlainJSON(string $filepath, $default = null)
    {
        if (!file_exists($filepath)) {
            return $default;
        }

        if (!is_readable($filepath)) {
            throw new \Exception("File is not readable: $filepath");
        }

        $contents = file_get_contents($filepath);
        if ($contents === false) {
            throw new \Exception("Failed to read file: $filepath");
        }

        if ($contents === '') {
            return $default;
        }

        $data = json_decode($contents, true);
        if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception("JSON decode failed for: $filepath – " . json_last_error_msg());
        }

        return $data;
    }

    /**
     * Write a plain (unencrypted) JSON file (atomic write).
     *
     * @param string $filepath
     * @param mixed  $data
     * @return bool
     * @throws \Exception
     */
    private function writePlainJSON(string $filepath, $data): bool
    {
        $dir = dirname($filepath);
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0700, true) && !is_dir($dir)) {
                if (!mkdir($dir, 0755, true) && !is_dir($dir)) {
                    throw new \Exception("Failed to create directory: $dir");
                }
            }
        }

        $json = json_encode($data, JSON_PRETTY_PRINT);
        if ($json === false) {
            throw new \Exception("JSON encode failed: " . json_last_error_msg());
        }

        $tempFile = $filepath . '.tmp.' . uniqid();
        if (file_put_contents($tempFile, $json) === false) {
            throw new \Exception("Failed to write temporary file: $tempFile");
        }

        chmod($tempFile, 0644);

        if (!rename($tempFile, $filepath)) {
            unlink($tempFile);
            throw new \Exception("Failed to rename temporary file to: $filepath");
        }

        return true;
    }

    /**
     * Read the public PGP key catalog
     *
     * @return array Public key catalog data
     * @throws \Exception
     */
    public function readPublicPGPKeyCatalog(): array
    {
        return $this->read(
            $this->config['pgpkeys_public'],
            $this->config['defaults']['pgpkeys_public']
        );
    }

    /**
     * Write the public PGP key catalog
     *
     * @param array $data Catalog data
     * @return bool
     * @throws \Exception
     */
    public function writePublicPGPKeyCatalog(array $data): bool
    {
        return $this->write($this->config['pgpkeys_public'], $data);
    }

    /**
     * Read user notes
     *
     * @return array User notes data
     * @throws \Exception
     */
    public function readUserNotes(): array
    {
        return $this->read(
            $this->config['user_notes'],
            $this->config['defaults']['user_notes']
        );
    }

    /**
     * Write user notes
     *
     * @param array $data User notes data
     * @return bool
     * @throws \Exception
     */
    public function writeUserNotes(array $data): bool
    {
        return $this->write($this->config['user_notes'], $data);
    }

    /**
     * Read credential notes
     *
     * @return array Credential notes data
     * @throws \Exception
     */
    public function readCredentialNotes(): array
    {
        return $this->read(
            $this->config['credential_notes'],
            $this->config['defaults']['credential_notes']
        );
    }

    /**
     * Write credential notes
     *
     * @param array $data Credential notes data
     * @return bool
     * @throws \Exception
     */
    public function writeCredentialNotes(array $data): bool
    {
        return $this->write($this->config['credential_notes'], $data);
    }

    /**
     * Read a share file
     *
     * @param string $shareId Share ID
     * @return array|null Share data or null if not found
     * @throws \Exception
     */
    public function readShare(string $shareId): ?array
    {
        $filepath = $this->config['shares_dir'] . '/' . $shareId . '.json.enc';
        return $this->read($filepath, null);
    }

    /**
     * Write a share file
     *
     * @param string $shareId Share ID
     * @param array $data Share data
     * @return bool
     * @throws \Exception
     */
    public function writeShare(string $shareId, array $data): bool
    {
        $filepath = $this->config['shares_dir'] . '/' . $shareId . '.json.enc';
        return $this->write($filepath, $data);
    }

    /**
     * Delete a share file
     *
     * @param string $shareId Share ID
     * @return bool
     */
    public function deleteShare(string $shareId): bool
    {
        $filepath = $this->config['shares_dir'] . '/' . $shareId . '.json.enc';
        return $this->delete($filepath);
    }

    /**
     * List all shares
     *
     * @return array Array of share IDs
     */
    public function listShares(): array
    {
        $shares = [];
        $files = glob($this->config['shares_dir'] . '/*.json.enc');

        foreach ($files as $file) {
            $shareId = basename($file, '.json.enc');
            $shares[] = $shareId;
        }

        return $shares;
    }

    /**
     * Read a public share file
     *
     * @param string $token Share token
     * @return array|null Share data or null if not found
     * @throws \Exception
     */
    public function readPublicShare(string $token): ?array
    {
        $filepath = $this->config['public_shares_dir'] . '/' . $token . '.json.enc';
        return $this->read($filepath, null);
    }

    /**
     * Write a public share file
     *
     * @param string $token Share token
     * @param array $data Share data
     * @return bool
     * @throws \Exception
     */
    public function writePublicShare(string $token, array $data): bool
    {
        $filepath = $this->config['public_shares_dir'] . '/' . $token . '.json.enc';
        return $this->write($filepath, $data);
    }

    /**
     * Delete a public share file
     *
     * @param string $token Share token
     * @return bool
     */
    public function deletePublicShare(string $token): bool
    {
        $filepath = $this->config['public_shares_dir'] . '/' . $token . '.json.enc';
        return $this->delete($filepath);
    }

    /**
     * List all public share tokens
     *
     * @return array Array of tokens
     */
    public function listPublicShares(): array
    {
        $dir = $this->config['public_shares_dir'] ?? '';
        if (!$dir || !is_dir($dir)) {
            return [];
        }
        $files = glob($dir . '/*.json.enc');
        if (!$files) {
            return [];
        }
        $tokens = [];
        foreach ($files as $file) {
            $tokens[] = basename($file, '.json.enc');
        }
        return $tokens;
    }

    /**
     * Initialize database files if they don't exist
     *
     * @return void
     * @throws \Exception
     */
    public function initialize(): void
    {
        // Create main database files if they don't exist
        if (!$this->exists($this->config['users'])) {
            $this->writeUsers($this->config['defaults']['users']);
        }

        if (!$this->exists($this->config['groups'])) {
            $this->writeGroups($this->config['defaults']['groups']);
        }

        if (!$this->exists($this->config['invites'])) {
            $this->writeInvites($this->config['defaults']['invites']);
        }

        if (!$this->exists($this->config['user_notes'])) {
            $this->writeUserNotes($this->config['defaults']['user_notes']);
        }

        if (!$this->exists($this->config['credential_notes'])) {
            $this->writeCredentialNotes($this->config['defaults']['credential_notes']);
        }
    }

    /**
     * Get the base data directory path
     *
     * @return string
     */
    public function getDataPath(): string
    {
        return dirname($this->config['credentials_dir']);
    }
}
