<?php

namespace PassGram\Core;

/**
 * Config Class
 *
 * Loads and manages configuration files for PassGram.
 * Provides easy access to configuration values throughout the application.
 */
class Config
{
    private static ?Config $instance = null;
    private array $config = [];
    private string $configPath;

    /**
     * Private constructor (Singleton pattern)
     *
     * @param string $configPath Path to config directory
     */
    private function __construct(string $configPath)
    {
        $this->configPath = $configPath;
        $this->loadConfigs();
    }

    /**
     * Get Config instance
     *
     * @param string|null $configPath Path to config directory
     * @return Config
     */
    public static function getInstance(?string $configPath = null): Config
    {
        if (self::$instance === null) {
            if ($configPath === null) {
                $configPath = dirname(__DIR__, 2) . '/config';
            }
            self::$instance = new self($configPath);
        }
        return self::$instance;
    }

    /**
     * Load all configuration files
     *
     * @return void
     */
    private function loadConfigs(): void
    {
        // Load main config
        $this->config['app'] = $this->loadFile('config.php');

        // Load security config
        $this->config['security'] = $this->loadFile('security.php');

        // Load database config
        $this->config['database'] = $this->loadFile('database.php');
    }

    /**
     * Load a configuration file
     *
     * @param string $filename Configuration filename
     * @return array Configuration data
     * @throws \Exception
     */
    private function loadFile(string $filename): array
    {
        $filepath = $this->configPath . '/' . $filename;

        if (!file_exists($filepath)) {
            throw new \Exception("Configuration file not found: $filepath");
        }

        $config = require $filepath;

        if (!is_array($config)) {
            throw new \Exception("Configuration file must return an array: $filepath");
        }

        return $config;
    }

    /**
     * Get a configuration value using dot notation
     *
     * @param string $key Configuration key (e.g., 'app.name', 'security.password_min_length')
     * @param mixed $default Default value if key not found
     * @return mixed Configuration value
     */
    public function get(string $key, $default = null)
    {
        $keys = explode('.', $key);
        $value = $this->config;

        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                return $default;
            }
            $value = $value[$k];
        }

        return $value;
    }

    /**
     * Set a configuration value using dot notation
     *
     * @param string $key Configuration key
     * @param mixed $value Value to set
     * @return void
     */
    public function set(string $key, $value): void
    {
        $keys = explode('.', $key);
        $config = &$this->config;

        while (count($keys) > 1) {
            $k = array_shift($keys);
            if (!isset($config[$k]) || !is_array($config[$k])) {
                $config[$k] = [];
            }
            $config = &$config[$k];
        }

        $config[array_shift($keys)] = $value;
    }

    /**
     * Check if a configuration key exists
     *
     * @param string $key Configuration key
     * @return bool
     */
    public function has(string $key): bool
    {
        $keys = explode('.', $key);
        $value = $this->config;

        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                return false;
            }
            $value = $value[$k];
        }

        return true;
    }

    /**
     * Get all configuration
     *
     * @return array All configuration data
     */
    public function all(): array
    {
        return $this->config;
    }

    /**
     * Get app configuration
     *
     * @return array App configuration
     */
    public function app(): array
    {
        return $this->config['app'];
    }

    /**
     * Get security configuration
     *
     * @return array Security configuration
     */
    public function security(): array
    {
        return $this->config['security'];
    }

    /**
     * Get database configuration
     *
     * @return array Database configuration
     */
    public function database(): array
    {
        return $this->config['database'];
    }

    /**
     * Get master application key
     *
     * @return string Master application key
     * @throws \Exception
     */
    public function getMasterKey(): string
    {
        $key = $this->get('security.master_application_key');

        if (empty($key) || $key === 'REPLACE_WITH_GENERATED_KEY_DURING_INSTALLATION') {
            throw new \Exception('Master application key not configured. Please run install.php first.');
        }

        return $key;
    }

    /**
     * Validate that installation has been completed
     *
     * @return bool
     */
    public function isInstalled(): bool
    {
        try {
            $this->getMasterKey();
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}
