<?php

namespace PassGram\Controllers;

use PassGram\Core\Config;
use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Models\User;
use PassGram\Models\Group;
use PassGram\Models\Credential;
use PassGram\Models\Share;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

class ShareController
{
    private Config $config;
    private Auth $auth;
    private CSRF $csrf;
    private User $userModel;
    private Group $groupModel;
    private Credential $credentialModel;
    private Share $shareModel;
    private Logger $logger;

    public function __construct(
        Config $config,
        Auth $auth,
        CSRF $csrf,
        User $userModel,
        Group $groupModel,
        Credential $credentialModel,
        Share $shareModel,
        Logger $logger
    ) {
        $this->config = $config;
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->userModel = $userModel;
        $this->groupModel = $groupModel;
        $this->credentialModel = $credentialModel;
        $this->shareModel = $shareModel;
        $this->logger = $logger;

        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    /**
     * Show form to share a credential with a specific user.
     * Recipients = all members of the current user's groups (no PGP key required).
     */
    public function showShareForm(): void
    {
        $userId = $this->auth->getUserId();
        $user   = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_GET['id'] ?? '');

        $encryptionKey = substr($user['master_key_hash'], 0, 32);

        try {
            $credential = $this->credentialModel->findById($userId, $credentialId, $encryptionKey, true);

            if (!$credential) {
                Session::flash('error', 'Credential not found');
                header('Location: /index.php?page=credentials');
                exit;
            }

            // All members of the groups the current user belongs to (except self)
            $groups = $this->groupModel->getByUserId($userId);
            $potentialRecipients = [];
            foreach ($groups as $group) {
                $members = $this->userModel->getByGroupId($group['id']);
                foreach ($members as $member) {
                    if ($member['id'] !== $userId) {
                        $potentialRecipients[$member['id']] = $member;
                    }
                }
            }
            $potentialRecipients = array_values($potentialRecipients);

            $data = [
                'credential'  => $credential,
                'recipients'  => $potentialRecipients,
                'csrf_token'  => $this->csrf->getToken(),
                'error'       => Session::flash('error'),
            ];

            $this->render('credentials/share', $data);

        } catch (\Exception $e) {
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials');
            exit;
        }
    }

    /**
     * Handle POST from the share form.
     * Encrypts the credential snapshot with the recipient's AES-256-GCM key —
     * no PGP key pair required on either side.
     */
    public function create(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=shares');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=shares');
            exit;
        }

        $userId       = $this->auth->getUserId();
        $user         = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_POST['credential_id'] ?? '');
        $recipientId  = Sanitizer::clean($_POST['recipient_id']  ?? '');

        $ownerKey = substr($user['master_key_hash'], 0, 32);

        try {
            // Decrypt credential with owner's key
            $credential = $this->credentialModel->findById($userId, $credentialId, $ownerKey, true);
            if (!$credential) {
                throw new \Exception('Credential not found');
            }

            // Load recipient
            $recipient = $this->userModel->findById($recipientId);
            if (!$recipient) {
                throw new \Exception('Recipient user not found');
            }

            // Derive recipient's encryption key (same derivation used for their own credentials)
            $recipientKey = substr($recipient['master_key_hash'], 0, 32);

            $permissions = [
                'can_view'    => true,
                'can_edit'    => isset($_POST['can_edit'])    && $_POST['can_edit']    === '1',
                'can_reshare' => isset($_POST['can_reshare']) && $_POST['can_reshare'] === '1',
            ];

            $this->shareModel->create(
                $credentialId,
                $userId,
                $recipientId,
                $credential,
                $recipientKey,
                $permissions
            );

            $this->logger->credentialAccess(
                $userId, $credentialId,
                'shared with user ' . $recipient['username']
            );

            Session::flash('success', 'Credential shared with ' . $recipient['username'] . ' successfully');
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Share create error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=shares&action=share&id=' . urlencode($credentialId));
            exit;
        }
    }

    /**
     * Revoke a user-to-user share.
     */
    public function revoke(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=shares');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=shares');
            exit;
        }

        $userId  = $this->auth->getUserId();
        $shareId = Sanitizer::clean($_POST['share_id'] ?? '');

        // Optional: return to credential view after revoking from there
        $credentialId = Sanitizer::clean($_POST['credential_id'] ?? '');

        try {
            $this->shareModel->revoke($shareId, $userId);
            $this->logger->credentialAccess($userId, $credentialId, "revoked share $shareId");

            Session::flash('success', 'Share revoked successfully');

            if ($credentialId) {
                header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId) . '&_=' . time());
            } else {
                header('Location: /index.php?page=shares');
            }
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Share revoke error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=shares');
            exit;
        }
    }

    /**
     * View a credential shared with the current user.
     * Decrypted automatically using the user's own session key — no passphrase needed.
     */
    public function view(): void
    {
        $userId  = $this->auth->getUserId();
        $user    = $this->auth->getCurrentUser();
        $shareId = Sanitizer::clean($_GET['id'] ?? '');

        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');

        try {
            $encryptionKey = substr($user['master_key_hash'], 0, 32);
            $result = $this->shareModel->get($shareId, $userId, $encryptionKey);

            if (!$result) {
                throw new \Exception('Share not found or access denied');
            }

            $this->logger->credentialAccess(
                $userId, $result['share']['credential_id'],
                'viewed shared credential'
            );

            $owner = $this->userModel->findById($result['share']['owner_id']);

            $data = [
                'share'      => $result['share'],
                'credential' => $result['credential'],
                'owner'      => $owner,
                'csrf_token' => $this->csrf->getToken(),
            ];

            $this->render('shares/view', $data);

        } catch (\Exception $e) {
            $this->logger->error('Share view error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=shares');
            exit;
        }
    }

    /**
     * List all shares (created by me / shared with me).
     */
    public function index(): void
    {
        $userId = $this->auth->getUserId();

        $myShares     = $this->shareModel->getByOwnerId($userId);
        $sharedWithMe = $this->shareModel->getSharedWithUser($userId);

        // Build a username map for all referenced user IDs
        $allUsers = $this->userModel->getAll();
        $userMap  = [];
        foreach ($allUsers as $u) {
            $userMap[$u['id']] = $u['username'];
        }

        $data = [
            'my_shares'     => $myShares,
            'shared_with_me' => $sharedWithMe,
            'user_map'      => $userMap,
            'csrf_token'    => $this->csrf->getToken(),
            'success'       => Session::flash('success'),
            'error'         => Session::flash('error'),
        ];

        $this->render('shares/list', $data);
    }

    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
