<?php

namespace PassGram\Controllers;

use PassGram\Core\Config;
use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Models\Credential;
use PassGram\Models\PublicShare;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

/**
 * PublicShareController
 *
 * Handles creation and one-time viewing of public credential share links.
 * Viewing requires no authentication; creating requires an active session.
 */
class PublicShareController
{
    private Config $config;
    private Auth $auth;
    private CSRF $csrf;
    private Credential $credentialModel;
    private PublicShare $publicShareModel;
    private Logger $logger;

    public function __construct(
        Config $config,
        Auth $auth,
        CSRF $csrf,
        Credential $credentialModel,
        PublicShare $publicShareModel,
        Logger $logger
    ) {
        $this->config           = $config;
        $this->auth             = $auth;
        $this->csrf             = $csrf;
        $this->credentialModel  = $credentialModel;
        $this->publicShareModel = $publicShareModel;
        $this->logger           = $logger;
    }

    /**
     * Create a one-time public share link for a credential (auth required).
     * POSTed from the credential view page.
     */
    public function create(): void
    {
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=credentials');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=credentials');
            exit;
        }

        $userId       = $this->auth->getUserId();
        $user         = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_POST['credential_id'] ?? '');

        $encryptionKey = substr($user['master_key_hash'], 0, 32);

        try {
            $credential = $this->credentialModel->findById($userId, $credentialId, $encryptionKey, true);

            if (!$credential) {
                throw new \Exception('Credential not found');
            }

            // Only share fields that make sense publicly
            $shareableData = [
                'title'    => $credential['title']    ?? '',
                'type'     => $credential['type']     ?? '',
                'username' => $credential['username'] ?? '',
                'password' => $credential['password'] ?? '',
                'url'      => $credential['url']      ?? '',
                'notes'    => $credential['notes']    ?? '',
                'tags'     => $credential['tags']     ?? [],
            ];

            $share = $this->publicShareModel->create($userId, $credentialId, $shareableData);

            $this->logger->credentialAccess(
                $userId, $credentialId,
                'public share link created (token: ' . substr($share['token'], 0, 8) . '…)'
            );

            $baseUrl  = rtrim($this->config->get('base_url', ''), '/');
            $shareUrl = $baseUrl . '/index.php?page=public-share&token=' . urlencode($share['token']);

            Session::flash('public_share_url', $shareUrl);
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Public share create error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId));
            exit;
        }
    }

    /**
     * View a public share link (no auth required).
     * Burns the link on first access — subsequent visits show the expired page.
     */
    public function view(): void
    {
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');

        $token = $_GET['token'] ?? '';

        // Validate token format before touching the filesystem
        if (!preg_match('/^[a-f0-9]{64}$/', $token)) {
            $this->renderExpired();
            return;
        }

        $share = $this->publicShareModel->consume($token);

        if (!$share) {
            $this->renderExpired();
            return;
        }

        $this->logger->credentialAccess(
            $share['owner_id'],
            $share['credential_id'],
            'public share viewed (token: ' . substr($token, 0, 8) . '…)'
        );

        $data = [
            'share'      => $share,
            'credential' => $share['credential_data'],
        ];

        $this->renderPublic('public_shares/view', $data);
    }

    /**
     * List all public shares the current user has created (auth required).
     */
    public function listShares(): void
    {
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }

        $userId = $this->auth->getUserId();
        $shares = $this->publicShareModel->getByOwnerId($userId);

        $baseUrl = rtrim($this->config->get('base_url', ''), '/');

        $data = [
            'shares'     => $shares,
            'base_url'   => $baseUrl,
            'csrf_token' => $this->csrf->getToken(),
            'success'    => Session::flash('success'),
            'error'      => Session::flash('error'),
        ];

        $this->render('public_shares/list', $data);
    }

    /**
     * Revoke (delete) a public share link (auth required, owner only).
     */
    public function revoke(): void
    {
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=public-share&action=list');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=public-share&action=list');
            exit;
        }

        $userId = $this->auth->getUserId();
        $token  = $_POST['token'] ?? '';

        // Validate token format
        if (!preg_match('/^[a-f0-9]{64}$/', $token)) {
            Session::flash('error', 'Invalid token');
            header('Location: /index.php?page=public-share&action=list');
            exit;
        }

        try {
            $this->publicShareModel->delete($token, $userId);
            Session::flash('success', 'Public share link deleted');
        } catch (\Exception $e) {
            $this->logger->error('Public share revoke error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
        }

        header('Location: /index.php?page=public-share&action=list');
        exit;
    }

    private function renderExpired(): void
    {
        $data = ['title' => 'Link Expired'];
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/public_shares/expired.php';
        require __DIR__ . '/../Views/layouts/footer.php';
        exit;
    }

    private function renderPublic(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
        exit;
    }

    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
        exit;
    }
}
