<?php

namespace PassGram\Controllers;

use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Models\PGPKey;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

/**
 * PGPKeyController
 *
 * Handles public PGP key storage, viewing, and group sharing.
 * PGP keys can be imported, stored, and shared with groups
 * just like password credentials.
 */
class PGPKeyController
{
    private Auth $auth;
    private CSRF $csrf;
    private PGPKey $pgpKeyModel;
    private Logger $logger;
    private ?\PassGram\Models\Group $groupModel;

    public function __construct(
        Auth $auth,
        CSRF $csrf,
        PGPKey $pgpKeyModel,
        Logger $logger,
        ?\PassGram\Models\Group $groupModel = null
    ) {
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->pgpKeyModel = $pgpKeyModel;
        $this->logger = $logger;
        $this->groupModel = $groupModel;

        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    /**
     * List all accessible PGP keys
     */
    public function index(): void
    {
        $userId = $this->auth->getUserId();

        if ($this->groupModel) {
            $userGroups = $this->groupModel->getByUserId($userId);
            $userGroupIds = array_map(function($group) { return $group['id']; }, $userGroups);
            $keys = $this->pgpKeyModel->getAllAccessible($userId, $userGroupIds);
        } else {
            $keys = $this->pgpKeyModel->getAll($userId);
        }

        // Enrich owned keys with group names
        foreach ($keys as &$key) {
            if (isset($key['is_owner']) && $key['is_owner'] && !empty($key['shared_with_groups'])) {
                $groupNames = [];
                foreach ($key['shared_with_groups'] as $groupShare) {
                    if ($this->groupModel) {
                        $group = $this->groupModel->findById($groupShare['group_id']);
                        if ($group) {
                            $groupNames[] = $group['name'];
                        }
                    }
                }
                $key['shared_group_names'] = $groupNames;
            } else {
                $key['shared_group_names'] = [];
            }
        }

        $data = [
            'keys' => $keys,
            'csrf_token' => $this->csrf->getToken(),
            'success' => Session::flash('success'),
            'error' => Session::flash('error'),
        ];

        $this->render('pgpkeys/list', $data);
    }

    /**
     * Show create/import form
     */
    public function create(): void
    {
        $userId = $this->auth->getUserId();

        $userGroups = [];
        if ($this->groupModel) {
            $userGroups = $this->groupModel->getByUserId($userId);
        }

        $data = [
            'csrf_token' => $this->csrf->getToken(),
            'user_groups' => $userGroups,
        ];

        $this->render('pgpkeys/create', $data);
    }

    /**
     * Store new PGP key
     */
    public function store(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgpkeys&action=create');
            exit;
        }

        $userId = $this->auth->getUserId();

        try {
            // Resolve public key: uploaded file takes precedence over pasted text
            $publicKey = trim($_POST['public_key'] ?? '');
            if (isset($_FILES['public_key_file']) && $_FILES['public_key_file']['error'] === UPLOAD_ERR_OK) {
                $uploaded = file_get_contents($_FILES['public_key_file']['tmp_name']);
                if ($uploaded === false) {
                    throw new \Exception('Failed to read uploaded key file');
                }
                $publicKey = trim($uploaded);
            }

            $keyData = [
                'label' => Sanitizer::clean($_POST['label'] ?? ''),
                'owner_name' => Sanitizer::clean($_POST['owner_name'] ?? ''),
                'owner_email' => Sanitizer::clean($_POST['owner_email'] ?? ''),
                'public_key' => $publicKey,
                'notes' => Sanitizer::clean($_POST['notes'] ?? ''),
                'tags' => isset($_POST['tags']) ? explode(',', Sanitizer::clean($_POST['tags'])) : [],
            ];

            $keyData['tags'] = array_map('trim', $keyData['tags']);
            $keyData['tags'] = array_filter($keyData['tags']);
            $keyData['is_public'] = isset($_POST['is_public']) && $_POST['is_public'] === '1';

            $key = $this->pgpKeyModel->create($userId, $keyData);

            $this->logger->credentialAccess($userId, $key['id'], 'PGP key stored: ' . $key['label']);

            // Share with selected groups
            $groupIds = $_POST['group_ids'] ?? [];
            if (!empty($groupIds) && is_array($groupIds)) {
                foreach ($groupIds as $groupId) {
                    $groupId = Sanitizer::clean($groupId);
                    if (!empty($groupId)) {
                        $this->pgpKeyModel->shareWithGroup($userId, $key['id'], $groupId);
                    }
                }
            }

            Session::flash('success', 'PGP key stored successfully');
            header('Location: /index.php?page=pgpkeys');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('PGP key store error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys&action=create');
            exit;
        }
    }

    /**
     * View PGP key details
     */
    public function show(): void
    {
        $userId = $this->auth->getUserId();
        $keyId = Sanitizer::clean($_GET['id'] ?? '');

        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');

        try {
            $key = $this->pgpKeyModel->findById($userId, $keyId);
            $isOwner = true;

            // If not found in current user's store, check group-shared keys
            if (!$key && $this->groupModel) {
                $userGroups = $this->groupModel->getByUserId($userId);
                $userGroupIds = array_map(function($group) { return $group['id']; }, $userGroups);

                $result = $this->pgpKeyModel->findByIdAcrossUsers($keyId, $userGroupIds);

                if ($result) {
                    $key = $result['key'];
                    $key['owner_id'] = $result['owner_id'];
                    $key['shared_via_group'] = $result['shared_via_group'];
                    $isOwner = false;
                }
            }

            if (!$key) {
                Session::flash('error', 'PGP key not found');
                header('Location: /index.php?page=pgpkeys');
                exit;
            }

            $this->logger->credentialAccess($userId, $keyId, 'PGP key viewed');

            $userGroups = [];
            if ($this->groupModel) {
                $userGroups = $this->groupModel->getByUserId($userId);
            }

            if (!isset($key['shared_with_groups'])) {
                $key['shared_with_groups'] = [];
            }

            $data = [
                'key' => $key,
                'is_owner' => $isOwner,
                'user_groups' => $userGroups,
                'csrf_token' => $this->csrf->getToken(),
                'success' => Session::flash('success'),
                'error' => Session::flash('error'),
            ];

            $this->render('pgpkeys/view', $data);

        } catch (\Exception $e) {
            $this->logger->error('PGP key view error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys');
            exit;
        }
    }

    /**
     * Show edit form
     */
    public function edit(): void
    {
        $userId = $this->auth->getUserId();
        $keyId = Sanitizer::clean($_GET['id'] ?? '');

        try {
            $key = $this->pgpKeyModel->findById($userId, $keyId);

            if (!$key) {
                Session::flash('error', 'PGP key not found');
                header('Location: /index.php?page=pgpkeys');
                exit;
            }

            $data = [
                'key' => $key,
                'csrf_token' => $this->csrf->getToken(),
            ];

            $this->render('pgpkeys/edit', $data);

        } catch (\Exception $e) {
            $this->logger->error('PGP key edit error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys');
            exit;
        }
    }

    /**
     * Update PGP key
     */
    public function update(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        $userId = $this->auth->getUserId();
        $keyId = Sanitizer::clean($_POST['id'] ?? '');

        try {
            $keyData = [
                'label' => Sanitizer::clean($_POST['label'] ?? ''),
                'owner_name' => Sanitizer::clean($_POST['owner_name'] ?? ''),
                'owner_email' => Sanitizer::clean($_POST['owner_email'] ?? ''),
                'notes' => Sanitizer::clean($_POST['notes'] ?? ''),
                'tags' => isset($_POST['tags']) ? explode(',', Sanitizer::clean($_POST['tags'])) : [],
            ];

            // Only update public key if a new one is provided (file takes precedence)
            $publicKey = trim($_POST['public_key'] ?? '');
            if (isset($_FILES['public_key_file']) && $_FILES['public_key_file']['error'] === UPLOAD_ERR_OK) {
                $uploaded = file_get_contents($_FILES['public_key_file']['tmp_name']);
                if ($uploaded === false) {
                    throw new \Exception('Failed to read uploaded key file');
                }
                $publicKey = trim($uploaded);
            }
            if (!empty($publicKey)) {
                $keyData['public_key'] = $publicKey;
            }

            $keyData['tags'] = array_map('trim', $keyData['tags']);
            $keyData['tags'] = array_filter($keyData['tags']);
            $keyData['is_public'] = isset($_POST['is_public']) && $_POST['is_public'] === '1';

            $this->pgpKeyModel->update($userId, $keyId, $keyData);

            $this->logger->credentialAccess($userId, $keyId, 'PGP key updated');

            Session::flash('success', 'PGP key updated successfully');
            header('Location: /index.php?page=pgpkeys&action=view&id=' . urlencode($keyId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('PGP key update error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys&action=edit&id=' . urlencode($keyId));
            exit;
        }
    }

    /**
     * Delete PGP key
     */
    public function delete(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        $userId = $this->auth->getUserId();
        $keyId = Sanitizer::clean($_POST['id'] ?? '');

        try {
            $this->pgpKeyModel->delete($userId, $keyId);

            $this->logger->credentialAccess($userId, $keyId, 'PGP key deleted');

            Session::flash('success', 'PGP key deleted successfully');
            header('Location: /index.php?page=pgpkeys');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('PGP key delete error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys');
            exit;
        }
    }

    /**
     * Share PGP key with group
     */
    public function shareWithGroup(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        $userId = $this->auth->getUserId();
        $keyId = Sanitizer::clean($_POST['key_id'] ?? '');
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');

        try {
            $this->pgpKeyModel->shareWithGroup($userId, $keyId, $groupId);

            $this->logger->credentialAccess($userId, $keyId, "PGP key shared with group $groupId");

            Session::flash('success', 'PGP key shared with group successfully');
            header('Location: /index.php?page=pgpkeys&action=view&id=' . urlencode($keyId) . '&_=' . time());
            exit;

        } catch (\Exception $e) {
            $this->logger->error('PGP key share error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys&action=view&id=' . urlencode($keyId));
            exit;
        }
    }

    /**
     * Revoke group access to PGP key
     */
    public function revokeGroupShare(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgpkeys');
            exit;
        }

        $userId = $this->auth->getUserId();
        $keyId = Sanitizer::clean($_POST['key_id'] ?? '');
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');

        try {
            $this->pgpKeyModel->revokeGroupShare($userId, $keyId, $groupId);

            $this->logger->credentialAccess($userId, $keyId, "PGP key group $groupId access revoked");

            Session::flash('success', 'Group access revoked successfully');
            header('Location: /index.php?page=pgpkeys&action=view&id=' . urlencode($keyId) . '&_=' . time());
            exit;

        } catch (\Exception $e) {
            $this->logger->error('PGP key revoke error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgpkeys&action=view&id=' . urlencode($keyId));
            exit;
        }
    }

    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
