<?php

namespace PassGram\Controllers;

use PassGram\Core\Config;
use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Security\PGP;
use PassGram\Security\PGPEncryption;
use PassGram\Models\User;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

class PGPController
{
    private Config $config;
    private Auth $auth;
    private CSRF $csrf;
    private User $userModel;
    private Logger $logger;

    public function __construct(
        Config $config,
        Auth $auth,
        CSRF $csrf,
        User $userModel,
        Logger $logger
    ) {
        $this->config = $config;
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->userModel = $userModel;
        $this->logger = $logger;

        // Require authentication
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    // -------------------------------------------------------------------------
    // Key generation
    // -------------------------------------------------------------------------

    public function showGenerate(): void
    {
        $user = $this->auth->getCurrentUser();

        $data = [
            'user'       => $user,
            'csrf_token' => $this->csrf->getToken(),
            'error'      => Session::flash('error'),
        ];

        $this->render('pgp/generate', $data);
    }

    public function generate(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgp');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgp');
            exit;
        }

        $userId    = $this->auth->getUserId();
        $user      = $this->auth->getCurrentUser();
        $passphrase = $_POST['passphrase'] ?? '';
        $keyType   = Sanitizer::clean($_POST['key_type'] ?? 'RSA');
        $keySize   = (int)($_POST['key_size'] ?? 4096);
        $ecCurve   = Sanitizer::clean($_POST['ec_curve'] ?? 'secp384r1');

        if (!in_array($keyType, ['RSA', 'DSA', 'EC'])) {
            throw new \Exception('Invalid key type selected');
        }

        if (!in_array($keySize, [2048, 3072, 4096])) {
            throw new \Exception('Invalid key size selected');
        }

        if (!in_array($ecCurve, ['secp384r1', 'secp521r1', 'prime256v1'])) {
            throw new \Exception('Invalid elliptic curve selected');
        }

        try {
            $keyPair = PGP::generateKeyPair($user['email'], $passphrase, $keySize, $keyType, $ecCurve);

            $pgpDir = $this->config->get('app.paths.pgp');
            if (!$pgpDir) {
                throw new \Exception('PGP directory not configured');
            }

            if (!is_dir($pgpDir)) {
                if (!mkdir($pgpDir, 0700, true)) {
                    throw new \Exception('Failed to create PGP directory');
                }
            }

            $privateKeyPath = $pgpDir . '/' . $userId . '_private.key.enc';
            $publicKeyPath  = $pgpDir . '/' . $userId . '_public.key';

            if (file_put_contents($privateKeyPath, $keyPair['private']) === false) {
                throw new \Exception('Failed to write private key file');
            }

            if (file_put_contents($publicKeyPath, $keyPair['public']) === false) {
                throw new \Exception('Failed to write public key file');
            }

            @chmod($privateKeyPath, 0600);
            @chmod($publicKeyPath, 0644);

            $this->userModel->setPGPKeyInfo(
                $userId,
                $keyPair['fingerprint'],
                $keyPair['algorithm'],
                $keyPair['key_size']
            );

            $this->logger->adminAction(
                $userId,
                'Generated ' . $keyPair['algorithm'] . ' PGP key pair (' . $keyPair['key_size'] . ')'
            );

            Session::flash('success', 'PGP key pair generated successfully!');
            header('Location: /index.php?page=pgp&action=view');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('PGP generation error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=pgp&action=show-generate');
            exit;
        }
    }

    // -------------------------------------------------------------------------
    // View
    // -------------------------------------------------------------------------

    public function view(): void
    {
        $userId = $this->auth->getUserId();
        $user   = $this->auth->getCurrentUser();

        $publicKey   = null;
        $fingerprint = null;

        if ($user['has_pgp_key']) {
            $pgpDir = $this->config->get('app.paths.pgp');
            if ($pgpDir) {
                $publicKeyPath = $pgpDir . '/' . $userId . '_public.key';
                if (file_exists($publicKeyPath)) {
                    $publicKey   = file_get_contents($publicKeyPath);
                    $fingerprint = $user['pgp_fingerprint'];
                }
            }
        }

        $pgpModeEnabled   = $user['settings']['pgp_encryption_mode'] ?? false;
        $passphraseInSession = Session::has('pgp_passphrase');

        $data = [
            'user'                  => $user,
            'public_key'            => $publicKey,
            'fingerprint'           => $fingerprint,
            'pgp_mode_enabled'      => $pgpModeEnabled,
            'passphrase_in_session' => $passphraseInSession,
            'csrf_token'            => $this->csrf->getToken(),
            'success'               => Session::flash('success'),
            'error'                 => Session::flash('error'),
        ];

        $this->render('pgp/view', $data);
    }

    // -------------------------------------------------------------------------
    // Passphrase unlock (store in session for PGP operations)
    // -------------------------------------------------------------------------

    /**
     * POST: Verify and store the PGP passphrase in the session so that
     * PGP-encrypted files can be read/written during this session.
     */
    public function unlockPGP(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $userId     = $this->auth->getUserId();
        $user       = $this->auth->getCurrentUser();
        $passphrase = $_POST['passphrase'] ?? '';

        $pgpDir = $this->config->get('app.paths.pgp');
        if (!$pgpDir) {
            Session::flash('error', 'PGP directory not configured');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $privateKeyPath = $pgpDir . '/' . $userId . '_private.key.enc';
        if (!file_exists($privateKeyPath)) {
            Session::flash('error', 'No PGP private key found. Generate a key pair first.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $encryptedPrivateKey = file_get_contents($privateKeyPath);

        if (!PGPEncryption::verifyPassphrase($encryptedPrivateKey, $passphrase)) {
            $this->logger->error('PGP unlock failed for user ' . $userId . ' – wrong passphrase');
            Session::flash('error', 'Incorrect passphrase. Please try again.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        Session::set('pgp_passphrase', $passphrase);

        $this->logger->adminAction($userId, 'PGP passphrase unlocked for session');
        Session::flash('success', 'PGP key unlocked for this session.');
        header('Location: /index.php?page=pgp&action=view');
        exit;
    }

    /**
     * POST: Clear the PGP passphrase from the session.
     */
    public function lockPGP(): void
    {
        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        Session::remove('pgp_passphrase');
        Session::flash('success', 'PGP key locked.');
        header('Location: /index.php?page=pgp&action=view');
        exit;
    }

    // -------------------------------------------------------------------------
    // Encryption mode toggle + migration
    // -------------------------------------------------------------------------

    /**
     * POST: Toggle PGP encryption mode on or off.
     *
     * Enabling  → migrates existing AES files to PGP-encrypted files.
     * Disabling → migrates PGP-encrypted files back to AES files.
     *
     * Requires the PGP passphrase to be in the session.
     */
    public function toggleEncryption(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $userId = $this->auth->getUserId();
        $user   = $this->auth->getCurrentUser();

        $currentMode = $user['settings']['pgp_encryption_mode'] ?? false;
        $newMode     = !$currentMode;

        if (!$user['has_pgp_key']) {
            Session::flash('error', 'You must generate a PGP key pair before enabling PGP encryption.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        // Passphrase is required for both directions (needed to set context for migration)
        $passphrase = Session::get('pgp_passphrase', '');
        if ($passphrase === '') {
            Session::flash('error', 'Unlock your PGP key first by entering your passphrase below.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $pgpDir = $this->config->get('app.paths.pgp');
        if (!$pgpDir) {
            Session::flash('error', 'PGP directory not configured.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $publicKeyPath  = $pgpDir . '/' . $userId . '_public.key';
        $privateKeyPath = $pgpDir . '/' . $userId . '_private.key.enc';

        if (!file_exists($publicKeyPath) || !file_exists($privateKeyPath)) {
            Session::flash('error', 'PGP key files not found.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        $publicKey          = file_get_contents($publicKeyPath);
        $encryptedPrivateKey = file_get_contents($privateKeyPath);

        // Inject our Database instance (accessible via the app's DI container or global)
        global $db;
        if (!isset($db)) {
            Session::flash('error', 'Database not available.');
            header('Location: /index.php?page=pgp&action=view');
            exit;
        }

        try {
            if ($newMode) {
                // Enable PGP mode – set context BEFORE migration so write methods use PGP
                $db->setPGPContext($publicKey, $encryptedPrivateKey, $passphrase);
                $db->migrateCredentialsToPGP($userId);
                $db->migratePublicKeysToPGPMode($userId);
                $db->clearPGPContext();

                $this->userModel->setPGPEncryptionMode($userId, true);

                $this->logger->adminAction($userId, 'Switched to PGP encryption mode');
                Session::flash('success', 'PGP encryption mode enabled. Your credentials and public keys are now encrypted with your PGP key.');

            } else {
                // Disable PGP mode – set context so read methods can decrypt PGP files
                $db->setPGPContext($publicKey, $encryptedPrivateKey, $passphrase);
                $db->migrateCredentialsToAES($userId);
                $db->migratePublicKeysToAES($userId);
                $db->clearPGPContext();

                $this->userModel->setPGPEncryptionMode($userId, false);

                $this->logger->adminAction($userId, 'Switched back to AES encryption mode');
                Session::flash('success', 'Reverted to standard AES encryption mode.');
            }

        } catch (\Exception $e) {
            $this->logger->error('Encryption mode toggle failed: ' . $e->getMessage());
            Session::flash('error', 'Migration failed: ' . $e->getMessage());
        }

        header('Location: /index.php?page=pgp&action=view');
        exit;
    }

    // -------------------------------------------------------------------------
    // Helpers
    // -------------------------------------------------------------------------

    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
