<?php

namespace PassGram\Controllers;

use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Models\Group;
use PassGram\Models\Invite;
use PassGram\Helpers\Sanitizer;

class InviteController
{
    private Auth $auth;
    private CSRF $csrf;
    private Group $groupModel;
    private Invite $inviteModel;

    public function __construct(
        Auth $auth,
        CSRF $csrf,
        Group $groupModel,
        Invite $inviteModel
    ) {
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->groupModel = $groupModel;
        $this->inviteModel = $inviteModel;

        // Require authentication
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    public function index(): void
    {
        $userId = $this->auth->getUserId();
        $groups = $this->groupModel->getByUserId($userId);

        // Get invites for all user's groups
        $allInvites = [];
        foreach ($groups as $group) {
            $invites = $this->inviteModel->getByGroupId($group['id']);
            foreach ($invites as $invite) {
                $invite['group_name'] = $group['name'];
                $allInvites[] = $invite;
            }
        }

        $data = [
            'invites' => $allInvites,
            'groups' => $groups,
            'csrf_token' => $this->csrf->getToken(),
            'success' => Session::flash('success'),
            'error' => Session::flash('error'),
        ];

        $this->render('invites/list', $data);
    }

    public function generate(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=invites');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=invites');
            exit;
        }

        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');

        try {
            // Verify user is member of group
            if (!$this->groupModel->isMember($groupId, $userId)) {
                throw new \Exception('You do not have permission to generate invites for this group');
            }

            $invite = $this->inviteModel->generate($groupId, $userId);

            Session::flash('success', 'Invite code generated successfully');
            Session::set('new_invite_code', $invite['code']);

            header('Location: /index.php?page=invites');
            exit;

        } catch (\Exception $e) {
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=invites');
            exit;
        }
    }

    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
