<?php

namespace PassGram\Controllers;

use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Models\User;
use PassGram\Models\Group;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

class GroupController
{
    private Auth $auth;
    private CSRF $csrf;
    private User $userModel;
    private Group $groupModel;
    private Logger $logger;

    public function __construct(
        Auth $auth,
        CSRF $csrf,
        User $userModel,
        Group $groupModel,
        Logger $logger
    ) {
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->userModel = $userModel;
        $this->groupModel = $groupModel;
        $this->logger = $logger;

        // Require authentication
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    public function index(): void
    {
        $userId = $this->auth->getUserId();
        $groups = $this->groupModel->getByUserId($userId);

        $data = [
            'groups' => $groups,
            'csrf_token' => $this->csrf->getToken(),
            'success' => Session::flash('success'),
            'error' => Session::flash('error'),
        ];

        $this->render('groups/list', $data);
    }

    public function view(): void
    {
        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_GET['id'] ?? '');

        try {
            $group = $this->groupModel->findById($groupId);

            if (!$group || !$this->groupModel->isMember($groupId, $userId)) {
                Session::flash('error', 'Group not found or access denied');
                header('Location: /index.php?page=groups');
                exit;
            }

            // Get group members
            $members = $this->userModel->getByGroupId($groupId);

            $data = [
                'group' => $group,
                'members' => $members,
                'is_owner' => $this->groupModel->isOwner($groupId, $userId),
                'csrf_token' => $this->csrf->getToken(),
                'success' => Session::flash('success'),
                'error' => Session::flash('error'),
            ];

            $this->render('groups/view', $data);

        } catch (\Exception $e) {
            $this->logger->error('Group view error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups');
            exit;
        }
    }

    public function create(): void
    {
        $data = [
            'csrf_token' => $this->csrf->getToken(),
        ];

        $this->render('groups/create', $data);
    }

    public function store(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=groups');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=groups&action=create');
            exit;
        }

        $userId = $this->auth->getUserId();

        try {
            $groupData = [
                'name' => Sanitizer::clean($_POST['name'] ?? ''),
                'description' => Sanitizer::clean($_POST['description'] ?? ''),
                'created_by' => $userId,
                'owner_id' => $userId,
            ];

            $group = $this->groupModel->create($groupData);

            $this->logger->adminAction($userId, 'Created group: ' . $group['name']);

            Session::flash('success', 'Group created successfully');
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($group['id']));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Group creation error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups&action=create');
            exit;
        }
    }

    public function edit(): void
    {
        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_GET['id'] ?? '');

        try {
            $group = $this->groupModel->findById($groupId);

            if (!$group || !$this->groupModel->isOwner($groupId, $userId)) {
                Session::flash('error', 'Group not found or you do not have permission to edit it');
                header('Location: /index.php?page=groups');
                exit;
            }

            $data = [
                'group' => $group,
                'csrf_token' => $this->csrf->getToken(),
            ];

            $this->render('groups/edit', $data);

        } catch (\Exception $e) {
            $this->logger->error('Group edit error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups');
            exit;
        }
    }

    public function update(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=groups');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=groups');
            exit;
        }

        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_POST['id'] ?? '');

        try {
            if (!$this->groupModel->isOwner($groupId, $userId)) {
                throw new \Exception('You do not have permission to edit this group');
            }

            $updateData = [
                'name' => Sanitizer::clean($_POST['name'] ?? ''),
                'description' => Sanitizer::clean($_POST['description'] ?? ''),
            ];

            $this->groupModel->update($groupId, $updateData);

            $this->logger->adminAction($userId, 'Updated group: ' . $groupId);

            Session::flash('success', 'Group updated successfully');
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Group update error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups&action=edit&id=' . urlencode($groupId));
            exit;
        }
    }

    public function delete(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=groups');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=groups');
            exit;
        }

        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_POST['id'] ?? '');

        try {
            if (!$this->groupModel->isOwner($groupId, $userId)) {
                throw new \Exception('You do not have permission to delete this group');
            }

            $group = $this->groupModel->findById($groupId);
            $this->groupModel->delete($groupId);

            $this->logger->adminAction($userId, 'Deleted group: ' . $group['name']);

            Session::flash('success', 'Group deleted successfully');
            header('Location: /index.php?page=groups');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Group delete error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups');
            exit;
        }
    }

    public function addMember(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=groups');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=groups');
            exit;
        }

        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');
        $memberUsername = Sanitizer::clean($_POST['username'] ?? '');

        try {
            if (!$this->groupModel->isOwner($groupId, $userId)) {
                throw new \Exception('You do not have permission to add members to this group');
            }

            // Find user by username
            $member = $this->userModel->findByUsername($memberUsername);
            if (!$member) {
                throw new \Exception('User not found');
            }

            $this->groupModel->addMember($groupId, $member['id']);

            $this->logger->adminAction($userId, "Added member {$member['username']} to group $groupId");

            Session::flash('success', 'Member added successfully');
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Add member error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;
        }
    }

    public function removeMember(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=groups');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=groups');
            exit;
        }

        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');
        $memberId = Sanitizer::clean($_POST['member_id'] ?? '');

        try {
            if (!$this->groupModel->isOwner($groupId, $userId)) {
                throw new \Exception('You do not have permission to remove members from this group');
            }

            $group = $this->groupModel->findById($groupId);
            if ($memberId === $group['owner_id']) {
                throw new \Exception('Cannot remove the group owner. Transfer ownership first.');
            }

            $this->groupModel->removeMember($groupId, $memberId);

            $this->logger->adminAction($userId, "Removed member $memberId from group $groupId");

            Session::flash('success', 'Member removed successfully');
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Remove member error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;
        }
    }

    public function transferOwnership(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=groups');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=groups');
            exit;
        }

        $userId = $this->auth->getUserId();
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');
        $newOwnerId = Sanitizer::clean($_POST['new_owner_id'] ?? '');

        try {
            if (!$this->groupModel->isOwner($groupId, $userId)) {
                throw new \Exception('You do not have permission to transfer ownership of this group');
            }

            $this->groupModel->transferOwnership($groupId, $newOwnerId);

            $this->logger->adminAction($userId, "Transferred ownership of group $groupId to user $newOwnerId");

            Session::flash('success', 'Ownership transferred successfully');
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Transfer ownership error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=groups&action=view&id=' . urlencode($groupId));
            exit;
        }
    }

    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
