<?php

namespace PassGram\Controllers;

use PassGram\Core\Config;
use PassGram\Core\Session;
use PassGram\Core\Database;
use PassGram\Security\Auth;
use PassGram\Security\Encryption;
use PassGram\Models\User;
use PassGram\Models\Group;
use PassGram\Models\Credential;
use PassGram\Models\Share;

/**
 * DashboardController
 *
 * Main dashboard showing overview of credentials and activity
 */
class DashboardController
{
    private Database $db;
    private Auth $auth;
    private Encryption $encryption;

    public function __construct(
        Database $db,
        Auth $auth,
        Encryption $encryption
    ) {
        $this->db = $db;
        $this->auth = $auth;
        $this->encryption = $encryption;

        // Require authentication
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    /**
     * Show dashboard
     */
    public function index(): void
    {
        $userId = $this->auth->getUserId();
        $user = $this->auth->getCurrentUser();

        // Initialize models
        $config = Config::getInstance();
        $validator = new \PassGram\Helpers\Validator($config);

        $credentialModel = new Credential($this->db, $validator);
        $shareModel = new Share($this->db, $validator);
        $groupModel = new Group($this->db, $validator);

        // Get user's credentials (without decrypting)
        $credentials = $credentialModel->getAll($userId, false);

        // Get recent credentials (last 10)
        $recentCredentials = array_slice($credentials, -10);
        $recentCredentials = array_reverse($recentCredentials);

        // Get shared credentials
        $sharedWithMe = $shareModel->getSharedWithUser($userId);

        // Get user's groups
        $groups = $groupModel->getByUserId($userId);

        // Get statistics
        $stats = [
            'total_credentials' => count($credentials),
            'shared_with_me' => count($sharedWithMe),
            'groups' => count($groups),
            'has_pgp_key' => $user['has_pgp_key'] ?? false,
        ];

        $data = [
            'user' => $user,
            'stats' => $stats,
            'recent_credentials' => $recentCredentials,
            'shared_credentials' => $sharedWithMe,
            'groups' => $groups,
        ];

        $this->render('dashboard/index', $data);
    }

    /**
     * Render view
     */
    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
