<?php

namespace PassGram\Controllers;

use PassGram\Core\Session;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Models\Credential;
use PassGram\Models\Share;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

/**
 * CredentialController
 *
 * Handles credential CRUD operations
 */
class CredentialController
{
    private Auth $auth;
    private CSRF $csrf;
    private Credential $credentialModel;
    private Logger $logger;
    private ?\PassGram\Models\Group $groupModel;
    private ?\PassGram\Models\User $userModel;
    private ?Share $shareModel;

    public function __construct(
        Auth $auth,
        CSRF $csrf,
        Credential $credentialModel,
        Logger $logger,
        ?\PassGram\Models\Group $groupModel = null,
        ?\PassGram\Models\User $userModel = null,
        ?Share $shareModel = null
    ) {
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->credentialModel = $credentialModel;
        $this->logger = $logger;
        $this->groupModel = $groupModel;
        $this->userModel = $userModel;
        $this->shareModel = $shareModel;

        // Require authentication
        if (!$this->auth->check()) {
            header('Location: /login.php');
            exit;
        }
    }

    /**
     * List all credentials
     */
    public function index(): void
    {
        $userId = $this->auth->getUserId();

        // Get all accessible credentials (owned + group-shared)
        if ($this->groupModel) {
            $userGroups = $this->groupModel->getByUserId($userId);
            $userGroupIds = array_map(function($group) { return $group['id']; }, $userGroups);
            $credentials = $this->credentialModel->getAllAccessible($userId, $userGroupIds, false);
        } else {
            // Fallback to just user's own credentials
            $credentials = $this->credentialModel->getAll($userId, false);
        }

        // Enrich owned credentials with group names for display
        foreach ($credentials as &$cred) {
            if (isset($cred['is_owner']) && $cred['is_owner'] && !empty($cred['shared_with_groups'])) {
                $groupNames = [];
                foreach ($cred['shared_with_groups'] as $groupShare) {
                    if ($this->groupModel) {
                        $group = $this->groupModel->findById($groupShare['group_id']);
                        if ($group) {
                            $groupNames[] = $group['name'];
                        }
                    }
                }
                $cred['shared_group_names'] = $groupNames;
            } else {
                $cred['shared_group_names'] = [];
            }
        }

        $data = [
            'credentials' => $credentials,
            'csrf_token' => $this->csrf->getToken(),
            'success' => Session::flash('success'),
            'error' => Session::flash('error'),
        ];

        $this->render('credentials/list', $data);
    }

    /**
     * Show create form
     */
    public function create(): void
    {
        $userId = $this->auth->getUserId();

        $userGroups = [];
        if ($this->groupModel) {
            $userGroups = $this->groupModel->getByUserId($userId);
        }

        $data = [
            'csrf_token' => $this->csrf->getToken(),
            'user_groups' => $userGroups,
        ];

        $this->render('credentials/create', $data);
    }

    /**
     * Store new credential
     */
    public function store(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=credentials');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=credentials&action=create');
            exit;
        }

        $userId = $this->auth->getUserId();
        $user = $this->auth->getCurrentUser();

        // Get encryption key (derived from user's master key hash)
        $encryptionKey = $this->getUserEncryptionKey($user);

        try {
            $credentialData = [
                'title' => Sanitizer::clean($_POST['title'] ?? ''),
                'type' => Sanitizer::clean($_POST['type'] ?? 'password'),
                'username' => Sanitizer::clean($_POST['username'] ?? ''),
                'password' => $_POST['password'] ?? '',
                'url' => Sanitizer::url($_POST['url'] ?? ''),
                'notes' => Sanitizer::clean($_POST['notes'] ?? ''),
                'tags' => isset($_POST['tags']) ? explode(',', Sanitizer::clean($_POST['tags'])) : [],
                'folder' => Sanitizer::clean($_POST['folder'] ?? ''),
            ];

            // Clean up tags
            $credentialData['tags'] = array_map('trim', $credentialData['tags']);
            $credentialData['tags'] = array_filter($credentialData['tags']);

            $credential = $this->credentialModel->create($userId, $credentialData, $encryptionKey);

            $this->logger->credentialAccess($userId, $credential['id'], 'created');

            // Share with selected groups
            $groupIds = $_POST['group_ids'] ?? [];
            $groupPermission = Sanitizer::clean($_POST['group_permission'] ?? 'read');
            if (!empty($groupIds) && is_array($groupIds)) {
                foreach ($groupIds as $groupId) {
                    $groupId = Sanitizer::clean($groupId);
                    if (!empty($groupId)) {
                        $this->credentialModel->shareWithGroup($userId, $credential['id'], $groupId, $groupPermission);
                        $this->logger->credentialAccess($userId, $credential['id'], "shared with group $groupId ($groupPermission access)");
                    }
                }
            }

            Session::flash('success', 'Credential created successfully');
            header('Location: /index.php?page=credentials');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Credential creation error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials&action=create');
            exit;
        }
    }

    /**
     * Show credential details
     */
    public function show(): void
    {
        $userId = $this->auth->getUserId();
        $user = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_GET['id'] ?? '');

        // Prevent browser caching of credential details
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');

        $encryptionKey = $this->getUserEncryptionKey($user);

        try {
            // First try to find in the current user's credentials
            $credential = $this->credentialModel->findById($userId, $credentialId, $encryptionKey, true);
            $isOwner = true;
            $accessLevel = 'write';

            // If not found, search group-shared credentials from other users
            if (!$credential && $this->groupModel && $this->userModel) {
                $userGroups = $this->groupModel->getByUserId($userId);
                $userGroupIds = array_map(function($group) { return $group['id']; }, $userGroups);

                $result = $this->credentialModel->findByIdAcrossUsers($credentialId, $userGroupIds);

                if ($result) {
                    // Found in another user's file - decrypt with the owner's key
                    $owner = $this->userModel->findById($result['owner_id']);
                    if ($owner) {
                        $ownerEncryptionKey = $this->getUserEncryptionKey($owner);
                        $credential = $this->credentialModel->findById(
                            $result['owner_id'],
                            $credentialId,
                            $ownerEncryptionKey,
                            true
                        );
                        if ($credential) {
                            $credential['owner_id'] = $result['owner_id'];
                            $credential['access_level'] = $result['access_level'];
                            $credential['shared_via_group'] = $result['shared_via_group'];
                            $isOwner = false;
                            $accessLevel = $result['access_level'];
                        }
                    }
                }
            }

            if (!$credential) {
                Session::flash('error', 'Credential not found');
                header('Location: /index.php?page=credentials');
                exit;
            }

            $this->logger->credentialAccess($userId, $credentialId, 'viewed');

            // Get user's groups for sharing options
            $userGroups = [];
            if ($this->groupModel) {
                $userGroups = $this->groupModel->getByUserId($userId);
            }

            // Initialize shared_with_groups if not set (for backwards compatibility)
            if (!isset($credential['shared_with_groups'])) {
                $credential['shared_with_groups'] = [];
            }

            // User-to-user shares for this credential (owner only)
            $userShares = [];
            $shareUserMap = [];
            if ($isOwner && $this->shareModel) {
                $userShares = $this->shareModel->getByCredentialId($credentialId, $userId);
                // Build username map for recipients
                if ($this->userModel) {
                    foreach ($userShares as $s) {
                        $rid = $s['shared_with_user_id'];
                        if (!isset($shareUserMap[$rid])) {
                            $r = $this->userModel->findById($rid);
                            $shareUserMap[$rid] = $r ? $r['username'] : $rid;
                        }
                    }
                }
            }

            $data = [
                'credential'      => $credential,
                'user_groups'     => $userGroups,
                'is_owner'        => $isOwner,
                'access_level'    => $accessLevel,
                'user_shares'     => $userShares,
                'share_user_map'  => $shareUserMap,
                'csrf_token'      => $this->csrf->getToken(),
                'success'         => Session::flash('success'),
                'error'           => Session::flash('error'),
                'public_share_url' => Session::flash('public_share_url'),
            ];

            $this->render('credentials/view', $data);

        } catch (\Exception $e) {
            $this->logger->error('Credential view error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials');
            exit;
        }
    }

    /**
     * Show edit form
     */
    public function edit(): void
    {
        $userId = $this->auth->getUserId();
        $user = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_GET['id'] ?? '');

        $encryptionKey = $this->getUserEncryptionKey($user);

        try {
            $credential = $this->credentialModel->findById($userId, $credentialId, $encryptionKey, true);
            $ownerId = $userId;

            // If not found in current user's file, check group-shared credentials
            if (!$credential && $this->groupModel && $this->userModel) {
                $userGroups = $this->groupModel->getByUserId($userId);
                $userGroupIds = array_map(function($group) { return $group['id']; }, $userGroups);

                $result = $this->credentialModel->findByIdAcrossUsers($credentialId, $userGroupIds);

                if ($result && $result['access_level'] === 'write') {
                    $owner = $this->userModel->findById($result['owner_id']);
                    if ($owner) {
                        $ownerEncryptionKey = $this->getUserEncryptionKey($owner);
                        $credential = $this->credentialModel->findById(
                            $result['owner_id'],
                            $credentialId,
                            $ownerEncryptionKey,
                            true
                        );
                        $ownerId = $result['owner_id'];
                    }
                }
            }

            if (!$credential) {
                Session::flash('error', 'Credential not found');
                header('Location: /index.php?page=credentials');
                exit;
            }

            $data = [
                'credential' => $credential,
                'owner_id' => $ownerId,
                'csrf_token' => $this->csrf->getToken(),
            ];

            $this->render('credentials/edit', $data);

        } catch (\Exception $e) {
            $this->logger->error('Credential edit error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials');
            exit;
        }
    }

    /**
     * Update credential
     */
    public function update(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=credentials');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=credentials');
            exit;
        }

        $userId = $this->auth->getUserId();
        $user = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_POST['id'] ?? '');
        $ownerId = Sanitizer::clean($_POST['owner_id'] ?? $userId);

        // Determine the correct encryption key and owner
        if ($ownerId !== $userId && $this->userModel) {
            // Editing a group-shared credential - use owner's encryption key
            $owner = $this->userModel->findById($ownerId);
            if (!$owner) {
                Session::flash('error', 'Credential owner not found');
                header('Location: /index.php?page=credentials');
                exit;
            }
            $encryptionKey = $this->getUserEncryptionKey($owner);
        } else {
            $ownerId = $userId;
            $encryptionKey = $this->getUserEncryptionKey($user);
        }

        try {
            $credentialData = [
                'title' => Sanitizer::clean($_POST['title'] ?? ''),
                'type' => Sanitizer::clean($_POST['type'] ?? 'password'),
                'username' => Sanitizer::clean($_POST['username'] ?? ''),
                'url' => Sanitizer::url($_POST['url'] ?? ''),
                'notes' => Sanitizer::clean($_POST['notes'] ?? ''),
                'tags' => isset($_POST['tags']) ? explode(',', Sanitizer::clean($_POST['tags'])) : [],
                'folder' => Sanitizer::clean($_POST['folder'] ?? ''),
            ];

            // Only update password if provided
            if (!empty($_POST['password'])) {
                $credentialData['password'] = $_POST['password'];
            }

            // Clean up tags
            $credentialData['tags'] = array_map('trim', $credentialData['tags']);
            $credentialData['tags'] = array_filter($credentialData['tags']);

            $this->credentialModel->update($ownerId, $credentialId, $credentialData, $encryptionKey);

            $this->logger->credentialAccess($userId, $credentialId, 'updated');

            Session::flash('success', 'Credential updated successfully');
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId));
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Credential update error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials&action=edit&id=' . urlencode($credentialId));
            exit;
        }
    }

    /**
     * Delete credential
     */
    public function delete(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=credentials');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=credentials');
            exit;
        }

        $userId = $this->auth->getUserId();
        $credentialId = Sanitizer::clean($_POST['id'] ?? '');

        try {
            $this->credentialModel->delete($userId, $credentialId);

            $this->logger->credentialAccess($userId, $credentialId, 'deleted');

            Session::flash('success', 'Credential deleted successfully');
            header('Location: /index.php?page=credentials');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Credential delete error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials');
            exit;
        }
    }

    /**
     * Redirect to the user-share form for a credential.
     */
    public function share(): void
    {
        $credentialId = Sanitizer::clean($_GET['id'] ?? '');
        header('Location: /index.php?page=shares&action=share&id=' . urlencode($credentialId));
        exit;
    }

    /**
     * Share credential with group
     */
    public function shareWithGroup(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=credentials');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=credentials');
            exit;
        }

        $userId = $this->auth->getUserId();
        $credentialId = Sanitizer::clean($_POST['credential_id'] ?? '');
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');
        $permission = Sanitizer::clean($_POST['permission'] ?? 'read');

        try {
            $this->credentialModel->shareWithGroup($userId, $credentialId, $groupId, $permission);

            $this->logger->credentialAccess($userId, $credentialId, "shared with group $groupId ($permission access)");

            Session::flash('success', 'Credential shared with group successfully');
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId) . '&_=' . time());
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Group share error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId));
            exit;
        }
    }

    /**
     * Revoke group access to credential
     */
    public function revokeGroupShare(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /index.php?page=credentials');
            exit;
        }

        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token');
            header('Location: /index.php?page=credentials');
            exit;
        }

        $userId = $this->auth->getUserId();
        $credentialId = Sanitizer::clean($_POST['credential_id'] ?? '');
        $groupId = Sanitizer::clean($_POST['group_id'] ?? '');

        try {
            $this->credentialModel->revokeGroupShare($userId, $credentialId, $groupId);

            $this->logger->credentialAccess($userId, $credentialId, "revoked group $groupId access");

            Session::flash('success', 'Group access revoked successfully');
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId) . '&_=' . time());
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Revoke group share error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /index.php?page=credentials&action=view&id=' . urlencode($credentialId));
            exit;
        }
    }

    /**
     * Get decrypted password for clipboard copy (AJAX)
     */
    public function getPassword(): void
    {
        header('Content-Type: application/json');
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');

        $userId = $this->auth->getUserId();
        $user = $this->auth->getCurrentUser();
        $credentialId = Sanitizer::clean($_GET['id'] ?? '');

        if (!$credentialId) {
            echo json_encode(['error' => 'Missing credential ID']);
            exit;
        }

        try {
            $encryptionKey = $this->getUserEncryptionKey($user);

            // Try current user's credentials first
            $credential = $this->credentialModel->findById($userId, $credentialId, $encryptionKey, true);

            // If not found, check group-shared credentials
            if (!$credential && $this->groupModel && $this->userModel) {
                $userGroups = $this->groupModel->getByUserId($userId);
                $userGroupIds = array_map(function($group) { return $group['id']; }, $userGroups);

                $result = $this->credentialModel->findByIdAcrossUsers($credentialId, $userGroupIds);

                if ($result) {
                    $owner = $this->userModel->findById($result['owner_id']);
                    if ($owner) {
                        $ownerEncryptionKey = $this->getUserEncryptionKey($owner);
                        $credential = $this->credentialModel->findById(
                            $result['owner_id'],
                            $credentialId,
                            $ownerEncryptionKey,
                            true
                        );
                    }
                }
            }

            if (!$credential || empty($credential['password'])) {
                echo json_encode(['error' => 'Credential not found or no password set']);
                exit;
            }

            $this->logger->credentialAccess($userId, $credentialId, 'password copied');

            echo json_encode(['password' => $credential['password']]);
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Get password error: ' . $e->getMessage());
            echo json_encode(['error' => 'Failed to retrieve password']);
            exit;
        }
    }

    /**
     * Generate random password (AJAX)
     */
    public function generatePassword(): void
    {
        $length = (int)($_GET['length'] ?? 16);
        $includeSymbols = ($_GET['symbols'] ?? 'true') === 'true';

        $password = Credential::generatePassword($length, $includeSymbols);

        header('Content-Type: application/json');
        echo json_encode(['password' => $password]);
        exit;
    }

    /**
     * Get user's encryption key
     */
    private function getUserEncryptionKey(array $user): string
    {
        // In a real implementation, this would derive from the user's master password
        // For now, we'll use the master_key_hash as the encryption key
        return substr($user['master_key_hash'], 0, 32);
    }

    /**
     * Render view
     */
    private function render(string $view, array $data = []): void
    {
        extract($data);
        require __DIR__ . '/../Views/layouts/header.php';
        require __DIR__ . '/../Views/' . $view . '.php';
        require __DIR__ . '/../Views/layouts/footer.php';
    }
}
