<?php

namespace PassGram\Controllers;

use PassGram\Core\Config;
use PassGram\Core\Session;
use PassGram\Core\Database;
use PassGram\Security\Auth;
use PassGram\Security\CSRF;
use PassGram\Security\Encryption;
use PassGram\Models\User;
use PassGram\Models\Group;
use PassGram\Models\Invite;
use PassGram\Helpers\Validator;
use PassGram\Helpers\Sanitizer;
use PassGram\Helpers\Logger;

/**
 * AuthController
 *
 * Handles authentication: login, logout, registration
 */
class AuthController
{
    private Config $config;
    private Database $db;
    private Auth $auth;
    private CSRF $csrf;
    private User $userModel;
    private Group $groupModel;
    private Invite $inviteModel;
    private Validator $validator;
    private Logger $logger;

    public function __construct(
        Config $config,
        Database $db,
        Auth $auth,
        CSRF $csrf,
        User $userModel,
        Group $groupModel,
        Invite $inviteModel,
        Validator $validator,
        Logger $logger
    ) {
        $this->config = $config;
        $this->db = $db;
        $this->auth = $auth;
        $this->csrf = $csrf;
        $this->userModel = $userModel;
        $this->groupModel = $groupModel;
        $this->inviteModel = $inviteModel;
        $this->validator = $validator;
        $this->logger = $logger;
    }

    /**
     * Show login form
     */
    public function showLogin(): void
    {
        // Redirect if already logged in
        if ($this->auth->isLoggedIn()) {
            header('Location: /index.php?page=dashboard');
            exit;
        }

        $data = [
            'csrf_token' => $this->csrf->getToken(),
            'error' => Session::flash('error'),
            'success' => Session::flash('success'),
        ];

        $this->render('auth/login', $data);
    }

    /**
     * Process login
     */
    public function processLogin(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /login.php');
            exit;
        }

        // Validate CSRF token
        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token. Please try again.');
            header('Location: /login.php');
            exit;
        }

        $username = Sanitizer::trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';

        try {
            $user = $this->auth->login($username, $password);

            if ($user) {
                $this->logger->info("User logged in: $username");
                header('Location: /index.php?page=dashboard');
                exit;
            } else {
                Session::flash('error', 'Invalid username or password');
                header('Location: /login.php');
                exit;
            }

        } catch (\Exception $e) {
            $this->logger->error('Login error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /login.php');
            exit;
        }
    }

    /**
     * Show registration form
     */
    public function showRegister(): void
    {
        // Redirect if already logged in
        if ($this->auth->isLoggedIn()) {
            header('Location: /index.php?page=dashboard');
            exit;
        }

        $inviteCode = Sanitizer::trim($_GET['invite'] ?? '');

        $data = [
            'csrf_token' => $this->csrf->getToken(),
            'error' => Session::flash('error'),
            'invite_code' => $inviteCode,
        ];

        $this->render('auth/register', $data);
    }

    /**
     * Process registration
     */
    public function processRegister(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /register.php');
            exit;
        }

        // Validate CSRF token
        if (!$this->csrf->validatePost()) {
            Session::flash('error', 'Invalid security token. Please try again.');
            header('Location: /register.php');
            exit;
        }

        $username = Sanitizer::trim($_POST['username'] ?? '');
        $email = Sanitizer::email($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        $inviteCode = Sanitizer::trim($_POST['invite_code'] ?? '');

        try {
            // Validate invite code
            $invite = $this->inviteModel->validate($inviteCode);

            if (!$invite) {
                throw new \Exception('Invalid or expired invite code');
            }

            // Validate passwords match
            if ($password !== $confirmPassword) {
                throw new \Exception('Passwords do not match');
            }

            // Create user
            $user = $this->userModel->create([
                'username' => $username,
                'email' => $email,
                'password' => $password,
            ]);

            // Add user to invite's group
            $this->userModel->addToGroup($user['id'], $invite['group_id']);
            $this->groupModel->addMember($invite['group_id'], $user['id']);

            // Mark invite as used
            $this->inviteModel->use($inviteCode, $user['id']);

            $this->logger->info("New user registered: $username (via invite)");

            Session::flash('success', 'Registration successful! Please log in.');
            header('Location: /login.php');
            exit;

        } catch (\Exception $e) {
            $this->logger->error('Registration error: ' . $e->getMessage());
            Session::flash('error', $e->getMessage());
            header('Location: /register.php?invite=' . urlencode($inviteCode));
            exit;
        }
    }

    /**
     * Logout
     */
    public function logout(): void
    {
        $username = Session::get('username', 'unknown');
        $this->auth->logout();
        $this->logger->info("User logged out: $username");

        Session::flash('success', 'You have been logged out successfully.');
        header('Location: /login.php');
        exit;
    }

    /**
     * Render view
     */
    private function render(string $view, array $data = []): void
    {
        extract($data);
        $viewPath = __DIR__ . '/../Views/' . $view . '.php';

        if (file_exists($viewPath)) {
            require $viewPath;
        } else {
            die("View not found: $view");
        }
    }
}
