/**
 * PassGram v3.0 - JavaScript Functions
 *
 * Client-side functionality for PassGram
 */

// Copy text to clipboard
function copyToClipboard(text) {
    // Create temporary textarea
    const textarea = document.createElement('textarea');
    textarea.value = text;
    textarea.style.position = 'fixed';
    textarea.style.opacity = '0';
    document.body.appendChild(textarea);

    textarea.select();
    textarea.setSelectionRange(0, 99999); // For mobile devices

    try {
        document.execCommand('copy');
        showNotification('Copied to clipboard!', 'success');
    } catch (err) {
        showNotification('Failed to copy', 'error');
    }

    document.body.removeChild(textarea);
}

// Toggle password visibility
function togglePassword(fieldId) {
    const field = document.getElementById(fieldId);
    const button = event.target;

    if (field.type === 'password') {
        field.type = 'text';
        button.textContent = 'Hide';
    } else {
        field.type = 'password';
        button.textContent = 'Show';
    }
}

// Generate random password
function generatePassword() {
    const length = 16;
    const charset = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*()_+-=[]{}|;:,.<>?';
    let password = '';

    // Ensure at least one of each type
    password += getRandomChar('abcdefghijklmnopqrstuvwxyz'); // lowercase
    password += getRandomChar('ABCDEFGHIJKLMNOPQRSTUVWXYZ'); // uppercase
    password += getRandomChar('0123456789'); // number
    password += getRandomChar('!@#$%^&*()_+-=[]{}|;:,.<>?'); // special

    // Fill the rest
    for (let i = password.length; i < length; i++) {
        password += getRandomChar(charset);
    }

    // Shuffle the password
    password = shuffleString(password);

    // Set the password field
    const passwordField = document.getElementById('password');
    if (passwordField) {
        passwordField.type = 'text';
        passwordField.value = password;
        showNotification('Password generated!', 'success');
    }

    return password;
}

// Get random character from charset
function getRandomChar(charset) {
    const randomIndex = Math.floor(Math.random() * charset.length);
    return charset[randomIndex];
}

// Shuffle string
function shuffleString(str) {
    const arr = str.split('');
    for (let i = arr.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [arr[i], arr[j]] = [arr[j], arr[i]];
    }
    return arr.join('');
}

// Show notification
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = 'alert alert-' + type;
    notification.textContent = message;
    notification.style.position = 'fixed';
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.zIndex = '9999';
    notification.style.minWidth = '200px';
    notification.style.boxShadow = '3px 3px 0 rgba(0,0,0,0.5)';

    document.body.appendChild(notification);

    // Auto-remove after 3 seconds
    setTimeout(() => {
        notification.style.opacity = '0';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

// Confirm delete action
function confirmDelete(message) {
    return confirm(message || 'Are you sure you want to delete this? This action cannot be undone.');
}

// Auto-hide success/error messages after 5 seconds
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert-success, .alert-error');
    alerts.forEach(alert => {
        if (!alert.classList.contains('no-auto-hide')) {
            setTimeout(() => {
                alert.style.opacity = '0';
                setTimeout(() => {
                    alert.remove();
                }, 300);
            }, 5000);
        }
    });
});

// Form validation
function validatePasswordStrength(password) {
    const minLength = 12;
    const hasUppercase = /[A-Z]/.test(password);
    const hasLowercase = /[a-z]/.test(password);
    const hasNumber = /[0-9]/.test(password);
    const hasSpecial = /[^a-zA-Z0-9]/.test(password);

    if (password.length < minLength) {
        return { valid: false, message: `Password must be at least ${minLength} characters` };
    }

    if (!hasUppercase) {
        return { valid: false, message: 'Password must include uppercase letters' };
    }

    if (!hasLowercase) {
        return { valid: false, message: 'Password must include lowercase letters' };
    }

    if (!hasNumber) {
        return { valid: false, message: 'Password must include numbers' };
    }

    if (!hasSpecial) {
        return { valid: false, message: 'Password must include special characters' };
    }

    return { valid: true, message: 'Password strength: Strong' };
}

// Add password strength indicator on registration/password change forms
document.addEventListener('DOMContentLoaded', function() {
    const passwordFields = document.querySelectorAll('input[type="password"][name="password"]');

    passwordFields.forEach(field => {
        // Skip if it's the login form
        if (field.form && field.form.action.includes('login')) {
            return;
        }

        const indicator = document.createElement('div');
        indicator.style.marginTop = '5px';
        indicator.style.fontSize = '11px';

        field.parentNode.insertBefore(indicator, field.nextSibling);

        field.addEventListener('input', function() {
            if (this.value.length === 0) {
                indicator.textContent = '';
                return;
            }

            const result = validatePasswordStrength(this.value);
            indicator.textContent = result.message;
            indicator.style.color = result.valid ? 'green' : 'red';
        });
    });
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl+G or Cmd+G: Generate password (when password field is focused)
    if ((e.ctrlKey || e.metaKey) && e.key === 'g') {
        const passwordField = document.activeElement;
        if (passwordField && passwordField.type === 'password') {
            e.preventDefault();
            generatePassword();
        }
    }
});

// Prevent accidental navigation away from forms
let formModified = false;

document.addEventListener('DOMContentLoaded', function() {
    const forms = document.querySelectorAll('form');

    forms.forEach(form => {
        const inputs = form.querySelectorAll('input, textarea, select');

        inputs.forEach(input => {
            input.addEventListener('change', function() {
                formModified = true;
            });
        });

        form.addEventListener('submit', function() {
            formModified = false;
        });
    });
});

window.addEventListener('beforeunload', function(e) {
    if (formModified) {
        e.preventDefault();
        e.returnValue = '';
        return '';
    }
});

// Console welcome message
console.log('%cPassGram v3.0', 'font-size: 24px; font-weight: bold; color: #5D009D;');
console.log('%cSecure Password Manager with PGP Encryption', 'font-size: 14px; color: #666;');
console.log('%cOS/2 Warp 3.0 Interface Theme', 'font-size: 12px; color: #999;');
