<?php
/**
 * PHP Embed Security Test
 *
 * Creates test files with forbidden functions and validates they're blocked
 */

require_once 'app/core/plugins.php';

$test_dir = __DIR__ . '/php_embeds/';
$test_results = [];

// Test 1: File operation (file_get_contents)
$dangerous_code_1 = <<<'PHP'
<?php
// This should be blocked
$content = file_get_contents('/etc/passwd');
echo $content;
?>
PHP;

$test_file_1 = $test_dir . '_security_test_file_ops.php';
file_put_contents($test_file_1, $dangerous_code_1);
$violations_1 = scan_php_code_for_blacklisted_functions('_security_test_file_ops.php');
unlink($test_file_1);

$test_results[] = [
    'name' => 'File operations (file_get_contents)',
    'passed' => !empty($violations_1),
    'details' => !empty($violations_1) ? 'Detected: ' . implode(', ', array_column($violations_1, 'function')) : 'FAILED: Not detected'
];

// Test 2: Shell command (exec)
$dangerous_code_2 = <<<'PHP'
<?php
$output = exec('ls -la');
echo $output;
?>
PHP;

$test_file_2 = $test_dir . '_security_test_shell.php';
file_put_contents($test_file_2, $dangerous_code_2);
$violations_2 = scan_php_code_for_blacklisted_functions('_security_test_shell.php');
unlink($test_file_2);

$test_results[] = [
    'name' => 'Shell commands (exec)',
    'passed' => !empty($violations_2),
    'details' => !empty($violations_2) ? 'Detected: ' . implode(', ', array_column($violations_2, 'function')) : 'FAILED: Not detected'
];

// Test 3: Code evaluation (eval)
$dangerous_code_3 = <<<'PHP'
<?php
eval('echo "dangerous";');
?>
PHP;

$test_file_3 = $test_dir . '_security_test_eval.php';
file_put_contents($test_file_3, $dangerous_code_3);
$violations_3 = scan_php_code_for_blacklisted_functions('_security_test_eval.php');
unlink($test_file_3);

$test_results[] = [
    'name' => 'Code evaluation (eval)',
    'passed' => !empty($violations_3),
    'details' => !empty($violations_3) ? 'Detected: ' . implode(', ', array_column($violations_3, 'function')) : 'FAILED: Not detected'
];

// Test 4: Network operations (curl_exec)
$dangerous_code_4 = <<<'PHP'
<?php
$ch = curl_init('http://example.com');
curl_exec($ch);
?>
PHP;

$test_file_4 = $test_dir . '_security_test_curl.php';
file_put_contents($test_file_4, $dangerous_code_4);
$violations_4 = scan_php_code_for_blacklisted_functions('_security_test_curl.php');
unlink($test_file_4);

$test_results[] = [
    'name' => 'Network operations (curl_exec)',
    'passed' => !empty($violations_4),
    'details' => !empty($violations_4) ? 'Detected: ' . implode(', ', array_column($violations_4, 'function')) : 'FAILED: Not detected'
];

// Test 5: Database operations (mysqli_connect)
$dangerous_code_5 = <<<'PHP'
<?php
$conn = mysqli_connect('localhost', 'user', 'pass');
mysqli_query($conn, 'SELECT * FROM users');
?>
PHP;

$test_file_5 = $test_dir . '_security_test_db.php';
file_put_contents($test_file_5, $dangerous_code_5);
$violations_5 = scan_php_code_for_blacklisted_functions('_security_test_db.php');
unlink($test_file_5);

$test_results[] = [
    'name' => 'Database operations (mysqli_connect, mysqli_query)',
    'passed' => !empty($violations_5),
    'details' => !empty($violations_5) ? 'Detected: ' . implode(', ', array_column($violations_5, 'function')) : 'FAILED: Not detected'
];

// Test 6: Include/require
$dangerous_code_6 = <<<'PHP'
<?php
include('/etc/passwd');
require('/etc/hosts');
?>
PHP;

$test_file_6 = $test_dir . '_security_test_include.php';
file_put_contents($test_file_6, $dangerous_code_6);
$violations_6 = scan_php_code_for_blacklisted_functions('_security_test_include.php');
unlink($test_file_6);

$test_results[] = [
    'name' => 'Include/Require operations',
    'passed' => !empty($violations_6),
    'details' => !empty($violations_6) ? 'Detected: ' . implode(', ', array_column($violations_6, 'function')) : 'FAILED: Not detected'
];

// Test 7: Safe code (no violations)
$safe_code = <<<'PHP'
<?php
$result = 5 + 10;
echo "Result: " . $result;
$text = str_replace('a', 'b', 'abc');
?>
PHP;

$test_file_7 = $test_dir . '_security_test_safe.php';
file_put_contents($test_file_7, $safe_code);
$violations_7 = scan_php_code_for_blacklisted_functions('_security_test_safe.php');
unlink($test_file_7);

$test_results[] = [
    'name' => 'Safe code (should pass)',
    'passed' => empty($violations_7),
    'details' => empty($violations_7) ? 'No violations detected (correct)' : 'FAILED: False positive - ' . implode(', ', array_column($violations_7, 'function'))
];

// Output HTML results
echo "<!DOCTYPE html>\n<html>\n<head><title>PHP Security Tests</title>\n";
echo "<style>
    body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
    h1 { color: #333; border-bottom: 3px solid #c62828; padding-bottom: 10px; }
    .test { margin: 10px 0; padding: 15px; background: white; border-left: 4px solid #ccc; }
    .pass { border-left-color: #28a745; }
    .fail { border-left-color: #dc3545; }
    .test-name { font-weight: bold; font-size: 14px; }
    .pass .test-name::before { content: '✓ '; color: #28a745; }
    .fail .test-name::before { content: '✗ '; color: #dc3545; }
    .details { margin-top: 5px; font-size: 12px; color: #666; }
    .summary { margin-top: 30px; padding: 20px; background: white; border: 2px solid #c62828; }
    .summary h2 { margin-top: 0; color: #c62828; }
    .stat { font-size: 18px; font-weight: bold; margin: 10px 0; }
    .stat.pass { color: #28a745; }
    .stat.fail { color: #dc3545; }
    .info { background: #e3f2fd; padding: 15px; border-left: 4px solid #2196f3; margin: 20px 0; }
</style>\n";
echo "</head>\n<body>\n";
echo "<h1>PHP Embed Security Validation Tests</h1>\n";

echo "<div class='info'>";
echo "<strong>Purpose:</strong> Verify that the blacklist successfully detects and blocks dangerous PHP functions.";
echo "</div>";

$passed_count = 0;
$failed_count = 0;

foreach ($test_results as $result) {
    $class = $result['passed'] ? 'pass' : 'fail';
    echo '<div class="test ' . $class . '">';
    echo '<div class="test-name">' . htmlspecialchars($result['name']) . '</div>';
    echo '<div class="details">' . htmlspecialchars($result['details']) . '</div>';
    echo '</div>';

    if ($result['passed']) {
        $passed_count++;
    } else {
        $failed_count++;
    }
}

echo "<div class='summary'>\n";
echo "<h2>Security Test Summary</h2>\n";
echo "<div class='stat pass'>Passed: $passed_count</div>\n";
echo "<div class='stat fail'>Failed: $failed_count</div>\n";
echo "<div class='stat'>Total: " . ($passed_count + $failed_count) . "</div>\n";

if ($failed_count === 0) {
    echo "<p style='color: #28a745; font-weight: bold; font-size: 20px; margin-top: 20px;'>✓ All security tests passed! Dangerous functions are properly blocked.</p>\n";
} else {
    echo "<p style='color: #dc3545; font-weight: bold; font-size: 20px; margin-top: 20px;'>✗ Security issues detected! Some dangerous functions are not being blocked.</p>\n";
}

echo "</div>\n";

echo "<div class='info' style='margin-top: 30px;'>";
echo "<strong>Note:</strong> These tests create temporary PHP files with dangerous code, scan them, and immediately delete them. ";
echo "No dangerous code is actually executed. The scanner detects function calls via tokenization.";
echo "</div>";

echo "</body>\n</html>";
