<?php
/**
 * PHP Embed System Test Suite
 *
 * Tests all components of the PHP embed system
 */

require_once 'app/core/plugins.php';

echo "<!DOCTYPE html>\n<html>\n<head>\n<title>PHP Embed Tests</title>\n";
echo "<style>
    body { font-family: monospace; padding: 20px; background: #f5f5f5; }
    h1 { color: #333; border-bottom: 3px solid #0078d7; padding-bottom: 10px; }
    .test { margin: 10px 0; padding: 15px; background: white; border-left: 4px solid #ccc; }
    .pass { border-left-color: #28a745; }
    .fail { border-left-color: #dc3545; }
    .test-name { font-weight: bold; font-size: 14px; }
    .pass .test-name::before { content: '✓ '; color: #28a745; }
    .fail .test-name::before { content: '✗ '; color: #dc3545; }
    .details { margin-top: 5px; font-size: 12px; color: #666; }
    .summary { margin-top: 30px; padding: 20px; background: white; border: 2px solid #0078d7; }
    .summary h2 { margin-top: 0; color: #0078d7; }
    .stat { font-size: 18px; font-weight: bold; margin: 10px 0; }
    .stat.pass { color: #28a745; }
    .stat.fail { color: #dc3545; }
</style>\n";
echo "</head>\n<body>\n";
echo "<h1>PHP Embed System Tests</h1>\n";

$tests_passed = 0;
$tests_failed = 0;

function test_result($name, $passed, $details = '') {
    global $tests_passed, $tests_failed;

    $class = $passed ? 'pass' : 'fail';

    echo '<div class="test ' . $class . '">';
    echo '<div class="test-name">' . htmlspecialchars($name) . '</div>';

    if (!empty($details)) {
        echo '<div class="details">' . htmlspecialchars($details) . '</div>';
    }

    echo '</div>';

    if ($passed) {
        $tests_passed++;
    } else {
        $tests_failed++;
    }
}

// Test 1: Path sanitization - valid path
$safe_path = sanitize_php_embed_path('test.php');
test_result(
    'Test 1: Valid path sanitization',
    $safe_path === 'test.php',
    'Input: "test.php", Output: "' . $safe_path . '"'
);

// Test 2: Directory traversal blocked
$unsafe_path = sanitize_php_embed_path('../../../etc/passwd');
test_result(
    'Test 2: Directory traversal blocked',
    $unsafe_path === false,
    'Input: "../../../etc/passwd" should be rejected'
);

// Test 3: Invalid extension blocked
$bad_ext = sanitize_php_embed_path('test.txt');
test_result(
    'Test 3: Invalid extension blocked',
    $bad_ext === false,
    'Only .php extension allowed'
);

// Test 4: Protocol handler blocked
$protocol = sanitize_php_embed_path('php://filter/resource=test.php');
test_result(
    'Test 4: Protocol handler blocked',
    $protocol === false,
    'php:// protocol should be rejected'
);

// Test 5: Parameter parsing
$params = parse_php_embed_params('micro=WISDOM meso=LOVE width=800');
test_result(
    'Test 5: Parameter parsing',
    $params['micro'] === 'WISDOM' &&
    $params['meso'] === 'LOVE' &&
    $params['width'] === '800',
    'Parsed: ' . json_encode($params)
);

// Test 6: Empty parameter string
$empty_params = parse_php_embed_params('');
test_result(
    'Test 6: Empty parameter string',
    is_array($empty_params) && count($empty_params) === 0,
    'Should return empty array'
);

// Test 7: Function blacklist exists and comprehensive
$blacklist = get_php_function_blacklist();
test_result(
    'Test 7: Function blacklist loaded',
    count($blacklist) > 50 &&
    in_array('eval', $blacklist) &&
    in_array('exec', $blacklist) &&
    in_array('file_get_contents', $blacklist),
    count($blacklist) . ' functions blacklisted'
);

// Test 8: Shortcode extraction - single embed
$content = "Text before [php:test.php param=value] text after";
$embeds = extract_php_embeds_from_content($content);
test_result(
    'Test 8: Shortcode extraction (single)',
    count($embeds) === 1 &&
    $embeds[0]['filepath'] === 'test.php',
    'Extracted: ' . $embeds[0]['filepath']
);

// Test 9: Multiple embeds
$multi_content = "[php:first.php] text [php:second.php width=800]";
$multi_embeds = extract_php_embeds_from_content($multi_content);
test_result(
    'Test 9: Multiple embed extraction',
    count($multi_embeds) === 2 &&
    $multi_embeds[0]['filepath'] === 'first.php' &&
    $multi_embeds[1]['filepath'] === 'second.php',
    'Found ' . count($multi_embeds) . ' embeds'
);

// Test 10: File exists check (for example file)
$exists = php_embed_exists('fractal_pattern2.php');
test_result(
    'Test 10: File exists check',
    $exists === true,
    'fractal_pattern2.php should exist'
);

// Test 11: Nonexistent file check
$not_exists = php_embed_exists('nonexistent_file.php');
test_result(
    'Test 11: Nonexistent file check',
    $not_exists === false,
    'Should return false for missing files'
);

// Test 12: Parameter controls generation
$metadata = [
    'parameters' => [
        'test_param' => [
            'type' => 'select',
            'options' => ['opt1', 'opt2'],
            'default' => 'opt1',
            'label' => 'Test Parameter'
        ]
    ]
];
$controls = generate_php_param_controls($metadata, 'test-id', []);
test_result(
    'Test 12: Parameter controls generation',
    strpos($controls, '<select') !== false &&
    strpos($controls, 'opt1') !== false &&
    strpos($controls, 'test-id') !== false,
    'Controls HTML generated with select element'
);

// Test 13: Render embed - nonexistent file
$render = render_php_embed('nonexistent.php', []);
test_result(
    'Test 13: Missing file error handling',
    strpos($render, 'Error: PHP file not found') !== false,
    'Error message displayed correctly'
);

// Test 14: Metadata extraction from fractal_pattern2.php
$metadata_test = get_php_embed_metadata('fractal_pattern2.php');
test_result(
    'Test 14: Metadata extraction',
    isset($metadata_test['parameters']['micro']) &&
    $metadata_test['parameters']['micro']['type'] === 'select' &&
    count($metadata_test['parameters']) === 7,
    'Extracted ' . count($metadata_test['parameters']) . ' parameters'
);

// Test 15: Parse embeds in content
$test_content = "# Test\n\n[php:fractal_pattern2.php micro=WISDOM]\n\nMore text";
$parsed = parse_php_embeds($test_content);
test_result(
    'Test 15: Parse embeds in content',
    strpos($parsed, '<div class="bookgram-php-embed-wrapper') !== false &&
    strpos($parsed, 'fractal_pattern2.php') !== false,
    'Shortcode replaced with iframe wrapper'
);

// Summary
echo "\n<div class='summary'>\n";
echo "<h2>Test Summary</h2>\n";
echo "<div class='stat " . ($tests_failed === 0 ? 'pass' : 'fail') . "'>Passed: $tests_passed</div>\n";
echo "<div class='stat " . ($tests_failed > 0 ? 'fail' : 'pass') . "'>Failed: $tests_failed</div>\n";
echo "<div class='stat'>Total: " . ($tests_passed + $tests_failed) . "</div>\n";

if ($tests_failed === 0) {
    echo "<p style='color: #28a745; font-weight: bold; font-size: 20px; margin-top: 20px;'>✓ All tests passed!</p>\n";
} else {
    echo "<p style='color: #dc3545; font-weight: bold; font-size: 20px; margin-top: 20px;'>✗ Some tests failed. Review failures above.</p>\n";
}

echo "</div>\n";
echo "</body>\n</html>";
