<?php
/**
 * Test script for HTML embed functionality
 * Run this from command line: php test-html-embeds.php
 */

// Load the plugins.php file which contains our HTML embed functions
require_once __DIR__ . '/app/core/plugins.php';

echo "=== HTML Embed System Tests ===\n\n";

// Test 1: Basic shortcode parsing
echo "Test 1: Basic shortcode parsing\n";
echo "--------------------------------\n";
$content1 = "Here is an embed: [html:test-embed.html]";
$result1 = parse_html_embeds($content1);
echo "Input: $content1\n";
echo "Contains iframe: " . (strpos($result1, '<iframe') !== false ? 'YES' : 'NO') . "\n";
echo "Contains test-embed.html: " . (strpos($result1, 'test-embed.html') !== false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 2: Shortcode with parameters
echo "Test 2: Shortcode with width and height parameters\n";
echo "---------------------------------------------------\n";
$content2 = "[html:test-embed.html width=800 height=400]";
$result2 = parse_html_embeds($content2);
echo "Input: $content2\n";
echo "Contains width=\"800px\": " . (strpos($result2, 'width="800px"') !== false ? 'YES' : 'NO') . "\n";
echo "Contains height=\"400px\": " . (strpos($result2, 'height="400px"') !== false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 3: Multiple embeds
echo "Test 3: Multiple embeds on same page\n";
echo "-------------------------------------\n";
$content3 = "[html:test-embed.html]\n\nSome text here.\n\n[html:AtomDiagramTool_Speed.html width=900]";
$result3 = parse_html_embeds($content3);
$iframe_count = substr_count($result3, '<iframe');
echo "Input: Two shortcodes\n";
echo "Number of iframes generated: $iframe_count\n";
echo "Expected: 2\n";
echo "Result: " . ($iframe_count === 2 ? 'PASS' : 'FAIL') . "\n";
echo "\n";

// Test 4: Non-existent file (error handling)
echo "Test 4: Error handling for missing file\n";
echo "----------------------------------------\n";
$content4 = "[html:nonexistent.html]";
$result4 = parse_html_embeds($content4);
echo "Input: $content4\n";
echo "Contains error message: " . (strpos($result4, 'html-embed-error') !== false ? 'YES' : 'NO') . "\n";
echo "Contains 'not found': " . (strpos($result4, 'not found') !== false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 5: Security - directory traversal attempt
echo "Test 5: Security - Directory traversal blocked\n";
echo "-----------------------------------------------\n";
$content5 = "[html:../../../etc/passwd]";
$result5 = parse_html_embeds($content5);
echo "Input: $content5\n";
echo "Contains error message: " . (strpos($result5, 'html-embed-error') !== false ? 'YES' : 'NO') . "\n";
echo "Does NOT contain iframe: " . (strpos($result5, '<iframe') === false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 6: Security - invalid extension
echo "Test 6: Security - Invalid file extension blocked\n";
echo "---------------------------------------------------\n";
$content6 = "[html:malicious.php]";
$result6 = parse_html_embeds($content6);
echo "Input: $content6\n";
echo "Contains error message: " . (strpos($result6, 'Invalid file path') !== false ? 'YES' : 'NO') . "\n";
echo "Does NOT contain iframe: " . (strpos($result6, '<iframe') === false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 7: Custom CSS class
echo "Test 7: Custom CSS class parameter\n";
echo "-----------------------------------\n";
$content7 = "[html:test-embed.html class=centered-embed]";
$result7 = parse_html_embeds($content7);
echo "Input: $content7\n";
echo "Contains centered-embed class: " . (strpos($result7, 'centered-embed') !== false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 8: Sandbox parameter
echo "Test 8: Sandbox parameter\n";
echo "-------------------------\n";
$content8 = "[html:test-embed.html sandbox=strict]";
$result8 = parse_html_embeds($content8);
echo "Input: $content8\n";
echo "Contains 'allow-scripts' only: " . (strpos($result8, 'sandbox="allow-scripts"') !== false ? 'YES' : 'NO') . "\n";
echo "\n";

// Test 9: Extract function
echo "Test 9: Extract HTML embeds from content\n";
echo "-----------------------------------------\n";
$content9 = "Text [html:file1.html] more text [html:file2.html width=500]";
$embeds = extract_html_embeds_from_content($content9);
echo "Input: Content with 2 shortcodes\n";
echo "Number of embeds extracted: " . count($embeds) . "\n";
echo "Expected: 2\n";
echo "Result: " . (count($embeds) === 2 ? 'PASS' : 'FAIL') . "\n";
if (count($embeds) > 0) {
    echo "First embed filepath: " . $embeds[0]['filepath'] . "\n";
}
echo "\n";

// Summary
echo "=== Test Summary ===\n";
echo "All tests completed. Check results above.\n";
echo "If all tests show expected results, the implementation is working correctly!\n";
