<?php
/**
 * PHP Embed Admin Functions Test
 *
 * Tests upload validation and security scanning
 */

require_once 'app/core/functions.php';
require_once 'app/core/plugins.php';

echo "<!DOCTYPE html>\n<html>\n<head>\n<title>PHP Embed Admin Tests</title>\n";
echo "<style>
    body { font-family: monospace; padding: 20px; background: #f5f5f5; }
    h1 { color: #333; border-bottom: 3px solid #0078d7; padding-bottom: 10px; }
    .test { margin: 10px 0; padding: 15px; background: white; border-left: 4px solid #ccc; }
    .pass { border-left-color: #28a745; }
    .fail { border-left-color: #dc3545; }
    .test-name { font-weight: bold; font-size: 14px; }
    .pass .test-name::before { content: '✓ '; color: #28a745; }
    .fail .test-name::before { content: '✗ '; color: #dc3545; }
    .details { margin-top: 5px; font-size: 12px; color: #666; }
    .summary { margin-top: 30px; padding: 20px; background: white; border: 2px solid #0078d7; }
    .summary h2 { margin-top: 0; color: #0078d7; }
    .stat { font-size: 18px; font-weight: bold; margin: 10px 0; }
    .stat.pass { color: #28a745; }
    .stat.fail { color: #dc3545; }
</style>\n";
echo "</head>\n<body>\n";
echo "<h1>PHP Embed Admin Functions Tests</h1>\n";

$tests_passed = 0;
$tests_failed = 0;

function test_result($name, $passed, $details = '') {
    global $tests_passed, $tests_failed;

    $class = $passed ? 'pass' : 'fail';

    echo '<div class="test ' . $class . '">';
    echo '<div class="test-name">' . htmlspecialchars($name) . '</div>';

    if (!empty($details)) {
        echo '<div class="details">' . htmlspecialchars($details) . '</div>';
    }

    echo '</div>';

    if ($passed) {
        $tests_passed++;
    } else {
        $tests_failed++;
    }
}

// Test 1: get_php_embeds() function
$php_embeds = get_php_embeds();
test_result(
    'Test 1: get_php_embeds() returns array',
    is_array($php_embeds),
    'Found ' . count($php_embeds) . ' PHP embed files'
);

// Test 2: fractal_pattern2.php should be listed
$has_fractal = false;
foreach ($php_embeds as $embed) {
    if ($embed['filename'] === 'fractal_pattern2.php') {
        $has_fractal = true;
        break;
    }
}
test_result(
    'Test 2: fractal_pattern2.php is listed',
    $has_fractal,
    'Example file should be present'
);

// Test 3: Test blacklist function availability
$blacklist = get_php_function_blacklist();
test_result(
    'Test 3: Function blacklist available',
    is_array($blacklist) && count($blacklist) > 50,
    count($blacklist) . ' functions blacklisted'
);

// Test 4: Test safe code scanning
$safe_code = <<<'PHP'
<?php
$result = 5 + 10;
echo "Result: " . $result;
?>
PHP;

$temp_file = '/tmp/test_safe_code.php';
file_put_contents($temp_file, $safe_code);

$tokens = token_get_all($safe_code);
$violations = [];
$language_constructs = [
    T_EVAL => 'eval',
    T_INCLUDE => 'include',
    T_INCLUDE_ONCE => 'include_once',
    T_REQUIRE => 'require',
    T_REQUIRE_ONCE => 'require_once'
];

foreach ($tokens as $token) {
    if (!is_array($token)) continue;
    if (isset($language_constructs[$token[0]])) {
        $violations[] = $language_constructs[$token[0]];
    }
}

unlink($temp_file);

test_result(
    'Test 4: Safe code scanning',
    empty($violations),
    empty($violations) ? 'No violations in safe code' : 'False positive: ' . implode(', ', $violations)
);

// Test 5: Test dangerous code detection - eval
$dangerous_code_eval = <<<'PHP'
<?php
eval('echo "test";');
?>
PHP;

$tokens = token_get_all($dangerous_code_eval);
$found_eval = false;
foreach ($tokens as $token) {
    if (is_array($token) && $token[0] === T_EVAL) {
        $found_eval = true;
        break;
    }
}

test_result(
    'Test 5: Dangerous code detection - eval',
    $found_eval,
    $found_eval ? 'eval() detected' : 'eval() NOT detected (FAIL)'
);

// Test 6: Test dangerous code detection - file_get_contents
$dangerous_code_file = <<<'PHP'
<?php
$content = file_get_contents('/etc/passwd');
?>
PHP;

$tokens = token_get_all($dangerous_code_file);
$found_file_get_contents = false;
foreach ($tokens as $i => $token) {
    if (is_array($token) && $token[0] === T_STRING && $token[1] === 'file_get_contents') {
        $found_file_get_contents = true;
        break;
    }
}

test_result(
    'Test 6: Dangerous code detection - file_get_contents',
    $found_file_get_contents,
    $found_file_get_contents ? 'file_get_contents() detected' : 'file_get_contents() NOT detected (FAIL)'
);

// Test 7: Test delete_php_embed with invalid filename
$delete_result = delete_php_embed('../etc/passwd');
test_result(
    'Test 7: delete_php_embed blocks directory traversal',
    $delete_result === false,
    'Should reject paths with directory traversal'
);

// Test 8: Test delete_php_embed with wrong extension
$delete_result = delete_php_embed('test.txt');
test_result(
    'Test 8: delete_php_embed blocks wrong extension',
    $delete_result === false,
    'Should reject non-PHP files'
);

// Test 9: Test format_bytes function
$formatted = format_bytes(1536);
test_result(
    'Test 9: format_bytes function',
    $formatted === '1.5 KB',
    'Input: 1536 bytes, Output: ' . $formatted
);

// Test 10: Verify README and QUICK_START exist
$readme_exists = file_exists(__DIR__ . '/php_embeds/README.md');
$quickstart_exists = file_exists(__DIR__ . '/php_embeds/QUICK_START.md');
test_result(
    'Test 10: Documentation files exist',
    $readme_exists && $quickstart_exists,
    'README.md: ' . ($readme_exists ? 'YES' : 'NO') . ', QUICK_START.md: ' . ($quickstart_exists ? 'YES' : 'NO')
);

// Summary
echo "\n<div class='summary'>\n";
echo "<h2>Test Summary</h2>\n";
echo "<div class='stat " . ($tests_failed === 0 ? 'pass' : 'fail') . "'>Passed: $tests_passed</div>\n";
echo "<div class='stat " . ($tests_failed > 0 ? 'fail' : 'pass') . "'>Failed: $tests_failed</div>\n";
echo "<div class='stat'>Total: " . ($tests_passed + $tests_failed) . "</div>\n";

if ($tests_failed === 0) {
    echo "<p style='color: #28a745; font-weight: bold; font-size: 20px; margin-top: 20px;'>✓ All admin function tests passed!</p>\n";
    echo "<p style='margin-top: 20px;'><strong>Next Step:</strong> Access the admin panel at <code>/admin</code> to test file upload functionality.</p>\n";
} else {
    echo "<p style='color: #dc3545; font-weight: bold; font-size: 20px; margin-top: 20px;'>✗ Some tests failed. Review failures above.</p>\n";
}

echo "</div>\n";

echo "<div style='background: #e8f4f8; border-left: 4px solid #0078d7; padding: 15px; margin-top: 20px;'>";
echo "<strong>Manual Testing Steps:</strong>";
echo "<ol>";
echo "<li>Go to <code>/admin</code> (requires admin login)</li>";
echo "<li>Scroll to 'PHP Embed Management' section</li>";
echo "<li>Try uploading a safe PHP file</li>";
echo "<li>Try uploading a PHP file with <code>file_get_contents()</code> - should be rejected</li>";
echo "<li>Verify files are listed in the table</li>";
echo "<li>Test delete functionality</li>";
echo "</ol>";
echo "</div>";

echo "</body>\n</html>";
