<?php // templates/page.php

// $page, $content, $revision_id_to_load, $is_old_revision are passed from the router

// --- Plugin Shortcode Parsing with Protection ---
require_once __DIR__ . '/../app/core/plugins.php';

// First, extract and protect HTML embeds and plugin content BEFORE markdown parsing
$plugin_placeholders = [];
$plugin_counter = 0;

// Extract HTML embeds first
$content = preg_replace_callback(
    '/\[html:([^\]]+)\]/i',
    function($matches) use (&$plugin_placeholders, &$plugin_counter) {
        // Render HTML embed and store it
        $embed_output = parse_html_embeds($matches[0]);
        // Use custom HTML element as placeholder
        $placeholder = '<bookgram-html-embed data-id="' . $plugin_counter . '"></bookgram-html-embed>';
        $plugin_placeholders[$placeholder] = $embed_output;
        $plugin_counter++;
        return $placeholder;
    },
    $content
);

// Then extract PHP embeds
$content = preg_replace_callback(
    '/\[php:([^\]]+)\]/i',
    function($matches) use (&$plugin_placeholders, &$plugin_counter) {
        // Render PHP embed and store it
        $embed_output = parse_php_embeds($matches[0]);
        // Use custom HTML element as placeholder
        $placeholder = '<bookgram-php-embed data-id="' . $plugin_counter . '"></bookgram-php-embed>';
        $plugin_placeholders[$placeholder] = $embed_output;
        $plugin_counter++;
        return $placeholder;
    },
    $content
);

// Then extract plugin shortcodes
$content = preg_replace_callback(
    '/\[plugin:([a-z0-9_]+)\s+id=([a-z0-9_-]+)([^\]]*)\]/i',
    function($matches) use (&$plugin_placeholders, &$plugin_counter, $page) {
        // Render plugin and store it
        $plugin_output = parse_shortcodes($matches[0], $page['uuid']);
        // Use custom HTML element as placeholder
        $placeholder = '<bookgram-plugin data-id="' . $plugin_counter . '"></bookgram-plugin>';
        $plugin_placeholders[$placeholder] = $plugin_output;
        $plugin_counter++;
        return $placeholder;
    },
    $content
);
// --- END Plugin Shortcode Parsing ---

// Now parse markdown (placeholders will pass through)
$parser = new Parsedown();
$raw_html = $parser->text($content);

// --- UUID Link Rewriting ---
$uuid_index = load_uuid_index();
$processed_html = preg_replace_callback(
    '/<a href="uuid:([a-zA-Z0-9_.-]+)">/i',
    function ($matches) use ($uuid_index) {
        $uuid = $matches[1];
        if (isset($uuid_index[$uuid])) {
            $path = $uuid_index[$uuid];
            return '<a href="/page/' . e($path) . '">';
        } else {
            return '<a href="/page/link-not-found?uuid=' . e($uuid) . '" class="broken-link" title="Broken Link (UUID: ' . e($uuid) . ')">';
        }
    },
    $raw_html
);
// --- END UUID Link Rewriting ---

// Sanitize - includes plugin support tags (div, form, input, etc.) and bookgram-plugin/bookgram-html-embed/bookgram-php-embed placeholders
$safe_html = strip_tags(
    $processed_html,
    '<p><b><i><strong><em><ul><ol><li><h1><h2><h3><h4><h5><h6><br><img><a><code><pre><blockquote><hr><table><thead><tbody><tr><th><td><div><span><form><input><textarea><select><option><button><label><fieldset><legend><bookgram-plugin><bookgram-html-embed><bookgram-php-embed>'
);

// Restore plugin content after sanitization
foreach ($plugin_placeholders as $placeholder => $plugin_content) {
    $safe_html = str_replace($placeholder, $plugin_content, $safe_html);
}

// --- NEW: Construct Full URL for Copy ---
$base_url_copy = (isset($_SERVER['HTTPS']) ? "https" : "http") . "://" . ($_SERVER['HTTP_HOST'] ?? 'localhost');
$fullPageUrl = $base_url_copy . '/page/' . ($page['path'] ?? '');
// --- END NEW ---

?>

<div class="window">
    <div class="window-title-bar">
        <span class="title"><?php echo e($page['title'] ?? 'Page Not Found'); ?></span>
    </div>
    <div class="window-content">

        <!-- Flash Messages -->
        <?php if (isset($_SESSION['admin_message'])): ?>
        <div style="background: #e8f4e8; border: 2px solid #4CAF50; padding: 12px; margin-bottom: 15px; font-weight: bold;">
            ✓ <?php echo e($_SESSION['admin_message']); unset($_SESSION['admin_message']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['admin_error'])): ?>
        <div style="background: #fdecea; border: 2px solid #f44336; padding: 12px; margin-bottom: 15px; font-weight: bold;">
            ✗ <?php echo e($_SESSION['admin_error']); unset($_SESSION['admin_error']); ?>
        </div>
        <?php endif; ?>

        <?php if (!empty($page['is_private'])): ?>
            <div class="notice-box">
                <strong>🔒 This is a private page<?php if (!empty($page['password_hash'])): ?> with password protection<?php endif; ?>.</strong>
                <?php if (is_logged_in() && (is_admin() || get_current_username() === ($page['author'] ?? ''))): ?>
                    <br>Only you and people with the share link<?php if (!empty($page['password_hash'])): ?> and password<?php endif; ?> can view this page.
                    <?php if (!empty($page['share_token'])): ?>
                        <br><br>
                        <strong>Share Link:</strong><br>
                        <input type="text" readonly
                               id="private_share_link"
                               value="<?php echo e($base_url_copy . '/page/' . ($page['path'] ?? '') . '?token=' . $page['share_token']); ?>"
                               style="width: 90%; margin-top: 5px; font-size: 12px;"
                               onclick="this.select(); document.execCommand('copy'); alert('Share link copied to clipboard!');">
                        <br><small>Click to copy the link. <?php if (!empty($page['password_hash'])): ?>Recipients will also need the password to access.<?php else: ?>Share this link to give others access.<?php endif; ?></small>
                    <?php endif; ?>
                <?php else: ?>
                    <br>This page is only visible to authorized users.
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <?php if ($is_old_revision): ?>
            <div class="notice-box">
                <strong>You are viewing an old revision from <?php echo date('Y-m-d H:i', (int)$revision_id_to_load); ?>.</strong><br>
                <a href="/page/<?php echo e($page['path'] ?? ''); ?>">View the current, live version.</a>
            </div>
        <?php endif; ?>

        <div class="page-actions-bar">
            <?php
            // Calculate folder path from page path
            $page_path = $page['path'] ?? '';
            $folder_path = dirname($page_path);
            if ($folder_path === '.' || $folder_path === '') {
                $folder_path = ''; // Root folder
            }
            $back_url = '/?path=' . urlencode($folder_path);
            ?>
            <a href="<?php echo e($back_url); ?>" class="edit-button" style="float: left; margin-right: 10px;">← Back to Folder</a>
            <button onclick="window.print();" class="print-button">Print</button>
            <?php if (is_editor() && !$is_old_revision): ?>
                <a href="/edit/<?php echo e($page['path'] ?? ''); ?>" class="edit-button">Edit this Page</a>
            <?php endif; ?>
        </div>

        <div class="page-body">
            <?php echo $safe_html; ?>
        </div>

        <hr>
        <div class="page-meta">
            <?php if (!empty($page['hashtags']) && is_array($page['hashtags'])): ?>
            <p>
                <strong>Tags:</strong>
                <?php foreach ($page['hashtags'] as $tag): ?>
                    <a href="/search?q=%23<?php echo urlencode($tag); ?>"
                       style="display: inline-block; background: #e0e0e0; padding: 3px 8px; margin: 2px; text-decoration: none; color: #000080; border: 1px solid #999;"
                       onmouseover="this.style.background='#d0d0d0'"
                       onmouseout="this.style.background='#e0e0e0'">
                        #<?php echo e($tag); ?>
                    </a>
                <?php endforeach; ?>
            </p>
            <?php endif; ?>

            <?php if (!empty($page['author'])): ?>
            <p>
                Original Author: <a href="/profile/<?php echo e($page['author']); ?>"><?php echo e($page['author']); ?></a>
            </p>
            <?php endif; ?>

            <p>
                Views: <?php echo e($page['hits'] ?? 0); ?>
                (Unique: <?php echo e($page['unique_hits'] ?? 0); ?>)
            </p>

            <?php // --- MODIFIED: Clickable Page ID --- ?>
            <?php if (!empty($page['uuid'])): ?>
                <p>
                    Page ID (
                    <?php // --- Pass the full URL to copyPageId --- ?>
                    <a href="#"
                       onclick="copyPageId('<?php echo e($fullPageUrl); ?>', this); return false;"
                       title="Click to copy page URL"
                       class="page-id-link">Copy Link</a>):
                    <code><?php echo e($page['uuid']); ?></code> <?php // Still display the UUID for internal linking ?>
                    <span class="copy-feedback" style="margin-left: 5px; color: green; display: none;">Copied!</span>
                </p>
            <?php else: ?>
                 <p>Page ID: <code>N/A</code></p>
            <?php endif; ?>
            <?php // --- END MODIFICATION --- ?>


            <?php if (!empty($page['history']) && is_array($page['history'])): ?>
                <p>Page History (<?php echo count($page['history']); ?> revisions):</p>
                <ul>
                    <?php foreach ($page['history'] as $rev_id): ?>
                        <li>
                            <?php $date = date('Y-m-d H:i:s', (int)$rev_id); ?>
                            <?php if ($rev_id == $revision_id_to_load): ?>
                                <strong><?php echo $date; ?> (Viewing)</strong>
                            <?php elseif ($rev_id == ($page['current_revision_id'] ?? null)): ?>
                                <a href="/page/<?php echo e($page['path'] ?? ''); ?>"><?php echo $date; ?> (Current)</a>
                            <?php else: ?>
                                <a href="/page/<?php echo e($page['path'] ?? ''); ?>?rev=<?php echo $rev_id; ?>"><?php echo $date; ?> (Old)</a>
                            <?php endif; ?>
                            <?php if (is_admin() && $rev_id != ($page['current_revision_id'] ?? null)): ?>
                                <form action="/admin-action" method="POST" class="inline-form">
                                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                                    <input type="hidden" name="action" value="delete_revision">
                                    <input type="hidden" name="path" value="<?php echo e($page['path'] ?? ''); ?>">
                                    <input type="hidden" name="revision_id" value="<?php echo $rev_id; ?>">
                                    <button type="submit" class="delete-button-small" title="Delete Revision"
                                            onclick="return confirm('Are you sure you want to permanently delete this one revision?');">
                                        D
                                    </button>
                                </form>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php else: ?>
                <p>No revision history available.</p>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
function copyPageId(textToCopy, element) {
    navigator.clipboard.writeText(textToCopy).then(function() {
        const feedback = element.parentElement.querySelector('.copy-feedback'); // Find feedback within the <p>
        if (feedback) {
            feedback.style.display = 'inline';
            setTimeout(() => {
                feedback.style.display = 'none';
            }, 1500);
        }
    }).catch(function(err) {
        console.error('Failed to copy text: ', err);
        alert('Failed to copy URL. Please copy it manually.');
    });
}
</script>
