/**
 * Plugin Admin Modal System
 * Creates OS/2-themed modal dialogs for plugin configuration
 */

(function() {
    'use strict';

    // Create modal container on page load
    let modalContainer = null;
    let currentIframe = null;
    let isDragging = false;
    let dragStartX = 0;
    let dragStartY = 0;
    let modalStartX = 0;
    let modalStartY = 0;

    function createModal() {
        if (modalContainer) return;

        const html = `
            <div id="plugin-admin-modal-overlay" class="plugin-modal-overlay">
                <div id="plugin-admin-modal" class="plugin-modal-window">
                    <div class="plugin-modal-titlebar">
                        <span class="plugin-modal-title">Plugin Configuration</span>
                        <div class="plugin-modal-controls">
                            <button class="plugin-modal-btn plugin-modal-minimize" title="Minimize">_</button>
                            <button class="plugin-modal-btn plugin-modal-maximize" title="Maximize">□</button>
                            <button class="plugin-modal-btn plugin-modal-close" title="Close">×</button>
                        </div>
                    </div>
                    <div class="plugin-modal-content">
                        <div class="plugin-modal-loading">Loading plugin configuration...</div>
                        <iframe id="plugin-admin-iframe" frameborder="0"></iframe>
                    </div>
                </div>
            </div>
        `;

        const div = document.createElement('div');
        div.innerHTML = html;
        document.body.appendChild(div.firstElementChild);

        modalContainer = document.getElementById('plugin-admin-modal-overlay');
        currentIframe = document.getElementById('plugin-admin-iframe');

        setupModalEvents();
    }

    function setupModalEvents() {
        const modal = document.getElementById('plugin-admin-modal');
        const titlebar = modal.querySelector('.plugin-modal-titlebar');
        const closeBtn = modal.querySelector('.plugin-modal-close');
        const minimizeBtn = modal.querySelector('.plugin-modal-minimize');
        const maximizeBtn = modal.querySelector('.plugin-modal-maximize');
        const overlay = modalContainer;

        // Close handlers
        closeBtn.addEventListener('click', closeModal);
        overlay.addEventListener('click', function(e) {
            if (e.target === overlay) {
                closeModal();
            }
        });

        // Minimize handler (hide)
        minimizeBtn.addEventListener('click', function() {
            modal.style.display = 'none';
            setTimeout(() => modal.style.display = 'flex', 100); // Quick hide/show effect
        });

        // Maximize/restore handler
        let isMaximized = false;
        let savedPosition = { width: null, height: null, left: null, top: null };

        maximizeBtn.addEventListener('click', function() {
            if (!isMaximized) {
                // Save current position
                savedPosition = {
                    width: modal.style.width || '800px',
                    height: modal.style.height || '600px',
                    left: modal.style.left || '50%',
                    top: modal.style.top || '50%'
                };

                // Maximize
                modal.style.width = '95vw';
                modal.style.height = '90vh';
                modal.style.left = '50%';
                modal.style.top = '50%';
                modal.style.transform = 'translate(-50%, -50%)';
                maximizeBtn.textContent = '❐'; // Different icon for restore
                isMaximized = true;
            } else {
                // Restore
                modal.style.width = savedPosition.width;
                modal.style.height = savedPosition.height;
                modal.style.left = savedPosition.left;
                modal.style.top = savedPosition.top;
                maximizeBtn.textContent = '□';
                isMaximized = false;
            }
        });

        // Dragging functionality
        titlebar.addEventListener('mousedown', function(e) {
            if (e.target.closest('.plugin-modal-controls')) return; // Don't drag when clicking buttons

            isDragging = true;
            dragStartX = e.clientX;
            dragStartY = e.clientY;

            const rect = modal.getBoundingClientRect();
            modalStartX = rect.left;
            modalStartY = rect.top;

            titlebar.style.cursor = 'grabbing';
            e.preventDefault();
        });

        document.addEventListener('mousemove', function(e) {
            if (!isDragging) return;

            const deltaX = e.clientX - dragStartX;
            const deltaY = e.clientY - dragStartY;

            modal.style.left = (modalStartX + deltaX) + 'px';
            modal.style.top = (modalStartY + deltaY) + 'px';
            modal.style.transform = 'none'; // Remove centering transform when dragging
        });

        document.addEventListener('mouseup', function() {
            if (isDragging) {
                isDragging = false;
                titlebar.style.cursor = 'grab';
            }
        });

        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            if (!modalContainer.classList.contains('active')) return;

            if (e.key === 'Escape') {
                closeModal();
            }
        });

        // Listen for messages from iframe (for closing on save)
        window.addEventListener('message', function(e) {
            if (e.data === 'plugin-admin-saved' || e.data === 'plugin-admin-close') {
                closeModal();
                // Reload the parent page to show updated plugin
                setTimeout(() => window.location.reload(), 500);
            }
        });
    }

    function openModal(url, title) {
        createModal();

        const modal = document.getElementById('plugin-admin-modal');
        const titleSpan = modal.querySelector('.plugin-modal-title');
        const loadingDiv = modal.querySelector('.plugin-modal-loading');

        titleSpan.textContent = title || 'Plugin Configuration';
        loadingDiv.style.display = 'block';
        currentIframe.style.display = 'none';
        currentIframe.src = '';

        modalContainer.classList.add('active');
        document.body.style.overflow = 'hidden'; // Prevent background scrolling

        // Load iframe content
        currentIframe.onload = function() {
            loadingDiv.style.display = 'none';
            currentIframe.style.display = 'block';
        };

        currentIframe.src = url;
    }

    function closeModal() {
        if (!modalContainer) return;

        modalContainer.classList.remove('active');
        document.body.style.overflow = ''; // Restore scrolling

        // Clear iframe after animation
        setTimeout(() => {
            if (currentIframe) {
                currentIframe.src = 'about:blank';
            }
        }, 300);
    }

    // Global function to open plugin admin
    window.openPluginAdmin = function(pageUuid, pluginId, instanceId, pluginName) {
        const url = '/plugin-admin/' + pageUuid + '/' + pluginId + '/' + instanceId + '?modal=1';
        const title = (pluginName || 'Plugin') + ' Configuration';
        openModal(url, title);
    };

    // Global function to close plugin admin
    window.closePluginAdmin = function() {
        closeModal();
    };

    // Auto-setup on page load
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', createModal);
    } else {
        createModal();
    }
})();
