/**
 * PHP Embed Interactive Parameters
 * Handles dynamic parameter updates without page reload
 */

/**
 * Update PHP embed iframe with new parameter values
 * @param {string} embedId - The iframe ID
 * @param {Event} event - The form submit event
 * @returns {boolean} - Always returns false to prevent default form submission
 */
function updatePhpEmbed(embedId, event) {
    event.preventDefault();

    const iframe = document.getElementById(embedId);
    if (!iframe) {
        console.error('Iframe not found:', embedId);
        return false;
    }

    const form = event.target;
    const formData = new FormData(form);

    // Get current iframe src
    const currentSrc = iframe.src;
    const url = new URL(currentSrc);

    // Update query parameters
    for (const [key, value] of formData.entries()) {
        url.searchParams.set(key, value);
    }

    // Reload iframe with new parameters
    iframe.src = url.toString();

    return false;
}

// Auto-update for range inputs with debouncing
document.addEventListener('DOMContentLoaded', function() {
    const rangeInputs = document.querySelectorAll('.php-param-form input[type="range"]');

    rangeInputs.forEach(input => {
        input.addEventListener('input', function() {
            // Debounce auto-update
            clearTimeout(this.updateTimeout);
            this.updateTimeout = setTimeout(() => {
                const form = this.closest('form');
                if (form) {
                    const submitEvent = new Event('submit', {
                        bubbles: true,
                        cancelable: true
                    });
                    form.dispatchEvent(submitEvent);
                }
            }, 500); // 500ms debounce - only update after user stops dragging for half a second
        });
    });

    // Optional: Add visual feedback when iframe is loading
    const iframes = document.querySelectorAll('.php-embed-iframe');
    iframes.forEach(iframe => {
        iframe.addEventListener('load', function() {
            const wrapper = this.closest('.bookgram-php-embed-wrapper');
            if (wrapper) {
                wrapper.classList.remove('loading');
            }
        });
    });
});
