// public/js/main.js

document.addEventListener('DOMContentLoaded', (event) => {
    console.log('DOM Content Loaded');

    const launchpad = document.getElementById('launchpad');

    // --- NEW: Load saved position from localStorage ---
    if (launchpad) {
        const savedTop = localStorage.getItem('launchpadTop');
        const savedLeft = localStorage.getItem('launchpadLeft');

        if (savedTop && savedLeft) {
            launchpad.style.top = savedTop;
            launchpad.style.left = savedLeft;
        }

        // Make the Launchpad draggable
        dragElement(launchpad);
    }
    // --- END NEW ---

    // Initialize hashtag autocomplete
    initHashtagAutocomplete();

    // Initialize hashtag cloud modal
    initHashtagCloud();
});

function dragElement(elmnt) {
    var pos1 = 0, pos2 = 0, pos3 = 0, pos4 = 0;
    
    // Use the launchpad's title bar to move it
    if (document.getElementById(elmnt.id + '-header')) {
        document.getElementById(elmnt.id + '-header').onmousedown = dragMouseDown;
    } else {
        // Fallback: drag from anywhere on the element
        elmnt.onmousedown = dragMouseDown;
    }

    function dragMouseDown(e) {
        e = e || window.event;
        e.preventDefault();
        // Get the mouse cursor position at startup
        pos3 = e.clientX;
        pos4 = e.clientY;
        document.onmouseup = closeDragElement;
        // Call a function whenever the cursor moves
        document.onmousemove = elementDrag;
    }

    function elementDrag(e) {
        e = e || window.event;
        e.preventDefault();
        // Calculate the new cursor position
        pos1 = pos3 - e.clientX;
        pos2 = pos4 - e.clientY;
        pos3 = e.clientX;
        pos4 = e.clientY;
        // Set the element's new position
        elmnt.style.top = (elmnt.offsetTop - pos2) + "px";
        elmnt.style.left = (elmnt.offsetLeft - pos1) + "px";
    }

    function closeDragElement() {
        // Stop moving when mouse button is released
        document.onmouseup = null;
        document.onmousemove = null;

        // --- NEW: Save the final position to localStorage ---
        localStorage.setItem('launchpadTop', elmnt.style.top);
        localStorage.setItem('launchpadLeft', elmnt.style.left);
        // --- END NEW ---
    }
}

// Hashtag autocomplete functionality
function initHashtagAutocomplete() {
    const searchInput = document.getElementById('search-input');
    if (!searchInput) return;

    const suggestionsDiv = document.getElementById('hashtag-suggestions');
    if (!suggestionsDiv) return;

    let debounceTimer;

    searchInput.addEventListener('input', function(e) {
        const value = e.target.value;

        // Clear previous timer
        clearTimeout(debounceTimer);

        // Check if value starts with #
        if (!value.startsWith('#') || value.length < 2) {
            suggestionsDiv.style.display = 'none';
            return;
        }

        // Debounce the API call
        debounceTimer = setTimeout(() => {
            const query = value.substring(1); // Remove the # symbol
            fetchHashtagSuggestions(query);
        }, 300);
    });

    function fetchHashtagSuggestions(query) {
        fetch('/api/hashtag-autocomplete?q=' + encodeURIComponent(query))
            .then(response => response.json())
            .then(suggestions => {
                if (suggestions.length === 0) {
                    suggestionsDiv.style.display = 'none';
                    return;
                }

                // Build suggestions list
                suggestionsDiv.innerHTML = '';
                suggestions.forEach(tag => {
                    const div = document.createElement('div');
                    div.textContent = '#' + tag;
                    div.style.padding = '5px 10px';
                    div.style.cursor = 'pointer';
                    div.style.borderBottom = '1px solid #ccc';

                    div.addEventListener('mouseenter', function() {
                        this.style.backgroundColor = '#e0e0e0';
                    });

                    div.addEventListener('mouseleave', function() {
                        this.style.backgroundColor = 'white';
                    });

                    div.addEventListener('click', function() {
                        searchInput.value = '#' + tag;
                        suggestionsDiv.style.display = 'none';
                        searchInput.form.submit();
                    });

                    suggestionsDiv.appendChild(div);
                });

                // Position and show suggestions
                const rect = searchInput.getBoundingClientRect();
                suggestionsDiv.style.top = (rect.bottom + window.scrollY) + 'px';
                suggestionsDiv.style.left = rect.left + 'px';
                suggestionsDiv.style.width = rect.width + 'px';
                suggestionsDiv.style.display = 'block';
            })
            .catch(error => {
                console.error('Error fetching hashtag suggestions:', error);
            });
    }

    // Hide suggestions when clicking outside
    document.addEventListener('click', function(e) {
        if (e.target !== searchInput && e.target.parentElement !== suggestionsDiv) {
            suggestionsDiv.style.display = 'none';
        }
    });
}

// Hashtag cloud modal functionality - Global functions for onclick handlers
function showHashtagCloudModal() {
    const modal = document.getElementById('hashtag-cloud-modal');
    const overlay = document.getElementById('hashtag-cloud-overlay');
    const content = document.getElementById('hashtag-cloud-content');

    if (!modal || !overlay || !content) {
        console.error('Hashtag cloud modal elements not found');
        return;
    }

    modal.style.display = 'block';
    overlay.style.display = 'block';
    loadTopHashtags();
}

function hideHashtagCloudModal() {
    const modal = document.getElementById('hashtag-cloud-modal');
    const overlay = document.getElementById('hashtag-cloud-overlay');

    if (modal) modal.style.display = 'none';
    if (overlay) overlay.style.display = 'none';
}

function loadTopHashtags() {
    const content = document.getElementById('hashtag-cloud-content');
    if (!content) return;

    content.innerHTML = 'Loading...';

    fetch('/api/top-hashtags?limit=100')
        .then(response => response.json())
        .then(hashtags => {
            if (Object.keys(hashtags).length === 0) {
                content.innerHTML = '<p>No hashtags found yet. Start using #hashtags in your pages!</p>';
                return;
            }

            // Create tag cloud with different sizes based on usage
            const maxCount = Math.max(...Object.values(hashtags));
            const minSize = 12;
            const maxSize = 32;

            let html = '<div style="line-height: 2.5;">';
            for (const [tag, count] of Object.entries(hashtags)) {
                // Calculate font size based on count
                const size = minSize + ((count / maxCount) * (maxSize - minSize));

                html += `<a href="/search?q=%23${encodeURIComponent(tag)}"
                            style="font-size: ${size}px;
                                   margin: 0 8px;
                                   text-decoration: none;
                                   color: #000080;
                                   display: inline-block;"
                            title="${count} page(s)"
                            onmouseover="this.style.textDecoration='underline'"
                            onmouseout="this.style.textDecoration='none'">
                            #${tag}
                         </a>`;
            }
            html += '</div>';

            content.innerHTML = html;
        })
        .catch(error => {
            console.error('Error loading hashtag cloud:', error);
            content.innerHTML = '<p>Error loading hashtags. Please try again.</p>';
        });
}

// Backwards compatibility - can be called from DOMContentLoaded
function initHashtagCloud() {
    // No longer needed - using onclick handlers instead
    console.log('Hashtag cloud initialized (using onclick handlers)');
}
