/**
 * TaskGram Plugin JavaScript
 * Handles AJAX interactions and client-side functionality
 */

// Global instance ID
let taskgramInstanceId = '';

/**
 * Initialize TaskGram
 */
function initTaskGram(instanceId) {
    taskgramInstanceId = instanceId;

    // Set up event listeners
    const addForm = document.getElementById('tg-add-form');
    if (addForm) {
        addForm.addEventListener('submit', handleAddTask);
    }

    const searchInput = document.getElementById('tg-search-input');
    if (searchInput) {
        searchInput.addEventListener('keyup', searchTasks);
    }

    const dueDateCheckbox = document.getElementById('tg-enable-due-date');
    if (dueDateCheckbox) {
        dueDateCheckbox.addEventListener('change', toggleDueDateFields);
        toggleDueDateFields(); // Initial state
    }

    const poolHeader = document.getElementById('tg-pool-header');
    if (poolHeader) {
        poolHeader.addEventListener('click', toggleCompletionPool);
    }
}

/**
 * Toggle due date/time fields visibility
 */
function toggleDueDateFields() {
    const checkbox = document.getElementById('tg-enable-due-date');
    const fields = document.getElementById('tg-due-datetime-fields');
    if (checkbox && fields) {
        fields.style.display = checkbox.checked ? 'flex' : 'none';
    }
}

/**
 * Handle add task form submission
 */
async function handleAddTask(e) {
    e.preventDefault();

    const form = e.target;
    const submitBtn = form.querySelector('button[type="submit"]');
    const text = form.querySelector('input[name="text"]').value.trim();

    if (!text) {
        alert('Please enter a task description');
        return;
    }

    // Disable button during submission
    submitBtn.disabled = true;
    submitBtn.textContent = 'Adding...';

    const formData = new FormData(form);
    formData.append('instance_id', taskgramInstanceId);
    formData.append('action', 'create_task');

    try {
        const response = await fetch('/plugin_taskgram/api/task.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + (result.message || 'Failed to add task'));
            submitBtn.disabled = false;
            submitBtn.textContent = 'Add Task';
        }
    } catch (error) {
        console.error('Error adding task:', error);
        alert('Error adding task. Please try again.');
        submitBtn.disabled = false;
        submitBtn.textContent = 'Add Task';
    }
}

/**
 * Complete a task (move to completion pool)
 */
async function completeTask(taskId) {
    if (!confirm('Mark this task as completed?')) {
        return;
    }

    const formData = new FormData();
    formData.append('instance_id', taskgramInstanceId);
    formData.append('action', 'complete_task');
    formData.append('task_id', taskId);

    try {
        const response = await fetch('/plugin_taskgram/api/task.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + (result.message || 'Failed to complete task'));
        }
    } catch (error) {
        console.error('Error completing task:', error);
        alert('Error completing task. Please try again.');
    }
}

/**
 * Delete a task permanently
 */
async function deleteTask(taskId) {
    if (!confirm('Delete this task permanently?')) {
        return;
    }

    const formData = new FormData();
    formData.append('instance_id', taskgramInstanceId);
    formData.append('action', 'delete_task');
    formData.append('task_id', taskId);

    try {
        const response = await fetch('/plugin_taskgram/api/task.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + (result.message || 'Failed to delete task'));
        }
    } catch (error) {
        console.error('Error deleting task:', error);
        alert('Error deleting task. Please try again.');
    }
}

/**
 * Restore a completed task to active
 */
async function restoreTask(taskId) {
    const formData = new FormData();
    formData.append('instance_id', taskgramInstanceId);
    formData.append('action', 'restore_task');
    formData.append('task_id', taskId);

    try {
        const response = await fetch('/plugin_taskgram/api/task.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + (result.message || 'Failed to restore task'));
        }
    } catch (error) {
        console.error('Error restoring task:', error);
        alert('Error restoring task. Please try again.');
    }
}

/**
 * Delete a completed task permanently
 */
async function deleteCompletedTask(taskId) {
    if (!confirm('Delete this completed task permanently?')) {
        return;
    }

    const formData = new FormData();
    formData.append('instance_id', taskgramInstanceId);
    formData.append('action', 'delete_completed');
    formData.append('task_id', taskId);

    try {
        const response = await fetch('/plugin_taskgram/api/task.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + (result.message || 'Failed to delete task'));
        }
    } catch (error) {
        console.error('Error deleting completed task:', error);
        alert('Error deleting task. Please try again.');
    }
}

/**
 * Clear entire completion pool
 */
async function clearCompletionPool() {
    if (!confirm('Delete ALL completed tasks? This cannot be undone.')) {
        return;
    }

    const formData = new FormData();
    formData.append('instance_id', taskgramInstanceId);
    formData.append('action', 'clear_completion_pool');

    try {
        const response = await fetch('/plugin_taskgram/api/task.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            alert('Deleted ' + (result.count || 0) + ' completed task(s)');
            location.reload();
        } else {
            alert('Error: ' + (result.message || 'Failed to clear completion pool'));
        }
    } catch (error) {
        console.error('Error clearing completion pool:', error);
        alert('Error clearing completion pool. Please try again.');
    }
}

/**
 * Search tasks (client-side filtering)
 */
function searchTasks() {
    const query = document.getElementById('tg-search-input').value.toLowerCase();
    const activeTasks = document.querySelectorAll('.tg-task[data-pool="active"]');
    const completedTasks = document.querySelectorAll('.tg-task[data-pool="completed"]');

    // Combine both lists
    const allTasks = [...activeTasks, ...completedTasks];

    if (query.length === 0) {
        // Show all tasks
        allTasks.forEach(task => {
            task.style.display = '';
        });
        return;
    }

    // Filter tasks
    allTasks.forEach(task => {
        const text = task.querySelector('.tg-task-text').textContent.toLowerCase();
        if (text.includes(query)) {
            task.style.display = '';
        } else {
            task.style.display = 'none';
        }
    });
}

/**
 * Toggle completion pool visibility
 */
function toggleCompletionPool() {
    const content = document.getElementById('tg-pool-content');
    const toggle = document.getElementById('tg-pool-toggle');

    if (content && toggle) {
        if (content.classList.contains('hidden')) {
            content.classList.remove('hidden');
            toggle.textContent = '▼';
        } else {
            content.classList.add('hidden');
            toggle.textContent = '▶';
        }
    }
}

/**
 * Edit task inline (future enhancement)
 */
function editTask(taskId) {
    // TODO: Implement inline editing
    alert('Inline editing coming soon! For now, you can delete and re-create the task.');
}
