<?php
/**
 * TaskGram Embed
 * Renders the task list interface
 */

// Get instance context from environment
$instance_path = $_ENV['BOOKGRAM_INSTANCE_PATH'] ?? null;
$instance_id = $_ENV['BOOKGRAM_INSTANCE_ID'] ?? 'unknown';
$page_uuid = $_ENV['BOOKGRAM_PAGE_UUID'] ?? null;

// Check if user is logged in and has editor/admin role
session_start();
$is_logged_in = isset($_SESSION['user']);
$can_edit = false;

if ($is_logged_in) {
    require_once __DIR__ . '/../app/core/auth.php';
    $can_edit = is_editor() || is_admin();
}

if (!$instance_path || !is_dir($instance_path)) {
    echo '<div class="taskgram-container">';
    echo '<p style="color: #cc0000;">Error: Instance path not found.</p>';
    echo '</div>';
    return;
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';
$tasks_file = $data_dir . 'tasks.json';
$completed_file = $data_dir . 'completed.json';

// Check if configured
if (!file_exists($config_file)) {
    echo '<div class="taskgram-container">';
    echo '<p>Task list not configured yet.</p>';
    echo '<p><a href="#" onclick="openPluginAdmin(\'' . htmlspecialchars($page_uuid) . '\', \'plugin_taskgram\', \'' . htmlspecialchars($instance_id) . '\', \'TaskGram\'); return false;">Configure Now</a></p>';
    echo '</div>';
    return;
}

// Load config
$config = json_decode(file_get_contents($config_file), true) ?? [];
$title = htmlspecialchars($config['title'] ?? 'Task List');
$description = htmlspecialchars($config['description'] ?? '');

// Load tasks
$tasks = [];
if (file_exists($tasks_file)) {
    $tasks = json_decode(file_get_contents($tasks_file), true) ?? [];
}

$completed_tasks = [];
if (file_exists($completed_file)) {
    $completed_tasks = json_decode(file_get_contents($completed_file), true) ?? [];
}

// Helper function to determine due date status
function get_due_status($due_date, $due_time) {
    if (empty($due_date)) {
        return null;
    }

    $now = new DateTime();
    $due = new DateTime($due_date);

    // If time is specified, include it
    if (!empty($due_time)) {
        $time_parts = explode(':', $due_time);
        $due->setTime((int)$time_parts[0], (int)$time_parts[1]);
    } else {
        // If no time, set to end of day
        $due->setTime(23, 59, 59);
    }

    if ($due < $now) {
        return 'overdue';
    } elseif ($due->format('Y-m-d') === $now->format('Y-m-d')) {
        return 'today';
    } else {
        return 'future';
    }
}

// Helper function to format due date display
function format_due_date($due_date, $due_time) {
    if (empty($due_date)) {
        return '';
    }

    $date = new DateTime($due_date);
    $formatted = $date->format('M j, Y');

    if (!empty($due_time)) {
        $time_parts = explode(':', $due_time);
        $hour = (int)$time_parts[0];
        $min = $time_parts[1];
        $ampm = $hour >= 12 ? 'PM' : 'AM';
        $hour_12 = $hour > 12 ? $hour - 12 : ($hour == 0 ? 12 : $hour);
        $formatted .= ' ' . $hour_12 . ':' . $min . ' ' . $ampm;
    }

    return $formatted;
}

?>
<link rel="stylesheet" href="/plugin_taskgram/css/style.css">
<div class="taskgram-container">
    <!-- Header -->
    <div class="tg-header tg-clearfix">
        <?php if ($can_edit): ?>
            <a href="#" class="tg-admin-link" onclick="openPluginAdmin('<?php echo htmlspecialchars($page_uuid); ?>', 'plugin_taskgram', '<?php echo htmlspecialchars($instance_id); ?>', 'TaskGram'); return false;">⚙ Configure</a>
        <?php endif; ?>
        <h3><?php echo $title; ?></h3>
        <?php if ($description): ?>
            <p><?php echo $description; ?></p>
        <?php endif; ?>
    </div>

    <!-- Search Bar -->
    <div class="tg-search">
        <input type="text" id="tg-search-input" placeholder="🔍 Search tasks..." autocomplete="off">
    </div>

    <?php if ($can_edit): ?>
    <!-- Add Task Form -->
    <div class="tg-add-task-form">
        <h4>Add New Task</h4>
        <form id="tg-add-form">
            <div class="tg-form-row">
                <label>Task Description:</label>
                <input type="text" name="text" required maxlength="500" placeholder="Enter task description...">
            </div>

            <div class="tg-form-row tg-form-inline">
                <input type="checkbox" id="tg-enable-due-date" name="enable_due_date">
                <label for="tg-enable-due-date">Set due date/time</label>
            </div>

            <div id="tg-due-datetime-fields" class="tg-datetime-inputs" style="display: none;">
                <div style="flex: 1;">
                    <label>Due Date:</label>
                    <input type="date" name="due_date">
                </div>
                <div style="flex: 1;">
                    <label>Due Time:</label>
                    <input type="time" name="due_time">
                </div>
            </div>

            <div class="tg-form-row" style="margin-top: 10px;">
                <button type="submit" class="primary">Add Task</button>
            </div>
        </form>
    </div>
    <?php else: ?>
    <!-- Read-only notice for guests -->
    <div style="padding: 10px; background: #f0f0f0; border: 2px solid #808080; margin-bottom: 20px; text-align: center;">
        <small>📋 Read-only view. <a href="/login">Log in</a> to add or modify tasks.</small>
    </div>
    <?php endif; ?>

    <!-- Active Tasks -->
    <div class="tg-tasks-section">
        <div class="tg-section-title">Active Tasks (<?php echo count($tasks); ?>)</div>

        <?php if (empty($tasks)): ?>
            <div class="tg-empty-state">
                No active tasks. Add one above!
            </div>
        <?php else: ?>
            <?php foreach ($tasks as $task): ?>
                <?php
                $task_id = htmlspecialchars($task['id']);
                $task_text = htmlspecialchars($task['text']);
                $due_date = $task['due_date'] ?? null;
                $due_time = $task['due_time'] ?? null;
                $due_status = get_due_status($due_date, $due_time);
                ?>
                <div class="tg-task" data-pool="active">
                    <?php if ($can_edit): ?>
                        <input type="checkbox" class="tg-task-checkbox" onclick="completeTask('<?php echo $task_id; ?>')">
                    <?php endif; ?>
                    <div class="tg-task-text"><?php echo $task_text; ?></div>

                    <?php if ($due_status): ?>
                        <div class="tg-due-badge <?php echo $due_status; ?>">
                            <?php echo format_due_date($due_date, $due_time); ?>
                        </div>
                    <?php endif; ?>

                    <?php if ($can_edit): ?>
                        <div class="tg-task-actions">
                            <button class="small danger" onclick="deleteTask('<?php echo $task_id; ?>')">Delete</button>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <!-- Completion Pool -->
    <?php if (!empty($completed_tasks)): ?>
        <div class="tg-completion-pool">
            <div class="tg-pool-header" id="tg-pool-header">
                <h4>Completed Tasks (<?php echo count($completed_tasks); ?>)</h4>
                <span class="tg-pool-toggle" id="tg-pool-toggle">▼</span>
            </div>

            <div class="tg-pool-content" id="tg-pool-content">
                <?php foreach ($completed_tasks as $task): ?>
                    <?php
                    $task_id = htmlspecialchars($task['id']);
                    $task_text = htmlspecialchars($task['text']);
                    $completed_at = $task['completed_at'] ?? '';
                    $due_date = $task['due_date'] ?? null;
                    $due_time = $task['due_time'] ?? null;
                    ?>
                    <div class="tg-completed-task tg-task" data-pool="completed">
                        <div class="tg-task-text" style="text-decoration: line-through; color: #666;">
                            <?php echo $task_text; ?>
                        </div>

                        <?php if ($due_date): ?>
                            <div class="tg-due-badge" style="background: #f0f0f0; color: #666; border-color: #999;">
                                <?php echo format_due_date($due_date, $due_time); ?>
                            </div>
                        <?php endif; ?>

                        <div class="tg-completed-meta">
                            Completed: <?php echo date('M j, Y g:i A', strtotime($completed_at)); ?>
                        </div>

                        <?php if ($can_edit): ?>
                            <div class="tg-task-actions">
                                <button class="small" onclick="restoreTask('<?php echo $task_id; ?>')">Restore</button>
                                <button class="small danger" onclick="deleteCompletedTask('<?php echo $task_id; ?>')">Delete</button>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>

                <?php if ($can_edit): ?>
                    <div class="tg-pool-actions">
                        <button class="danger" onclick="clearCompletionPool()">Clear All Completed Tasks</button>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>

    <!-- Export Section -->
    <div class="tg-export-section">
        <h4>Export</h4>
        <div class="tg-export-buttons">
            <?php if (!empty($tasks)): ?>
                <button onclick="window.location.href='/plugin_taskgram/api/export-csv.php?instance_id=<?php echo urlencode($instance_id); ?>&type=active'">
                    📄 Export Active Tasks (CSV)
                </button>
            <?php endif; ?>

            <?php if (!empty($completed_tasks)): ?>
                <button onclick="window.location.href='/plugin_taskgram/api/export-csv.php?instance_id=<?php echo urlencode($instance_id); ?>&type=completed'">
                    📄 Export Completed Tasks (CSV)
                </button>
            <?php endif; ?>

            <?php
            // Count tasks with due dates
            $tasks_with_dates = array_filter($tasks, function($task) {
                return !empty($task['due_date']);
            });
            ?>
            <?php if (!empty($tasks_with_dates)): ?>
                <button onclick="window.location.href='/plugin_taskgram/api/export-ical.php?instance_id=<?php echo urlencode($instance_id); ?>'">
                    📅 Export Due Dates (iCal)
                </button>
            <?php endif; ?>
        </div>
    </div>
</div>

<script src="/plugin_taskgram/js/taskgram.js"></script>
<script>
// Initialize TaskGram
initTaskGram('<?php echo htmlspecialchars($instance_id); ?>');
</script>
