<?php
/**
 * TaskGram API - Task Operations
 * Handles CRUD operations for tasks
 */

session_start();
header('Content-Type: application/json');

// Check authentication - require editor or admin role
if (!isset($_SESSION['user'])) {
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit;
}

// Load auth functions to check role
require_once __DIR__ . '/../../app/core/auth.php';

if (!is_editor() && !is_admin()) {
    echo json_encode(['success' => false, 'message' => 'Editor or admin role required']);
    exit;
}

// Helper function to find instance path
function find_instance_path($instance_id) {
    $base = __DIR__ . '/../../app/storage/plugin_instances/';

    if (!is_dir($base)) {
        return null;
    }

    // Iterate through page UUIDs
    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_taskgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }

    return null;
}

// Helper function to generate unique task ID
function generate_task_id() {
    return 'task_' . uniqid() . '_' . time();
}

// Sanitize instance ID
$instance_id = $_POST['instance_id'] ?? null;
if (!$instance_id) {
    echo json_encode(['success' => false, 'message' => 'Missing instance_id']);
    exit;
}

$instance_id = preg_replace('/[^a-zA-Z0-9_-]/', '', $instance_id);

// Find instance path
$instance_path = find_instance_path($instance_id);
if (!$instance_path) {
    echo json_encode(['success' => false, 'message' => 'Instance not found']);
    exit;
}

$data_dir = $instance_path . '/data/';
$tasks_file = $data_dir . 'tasks.json';
$completed_file = $data_dir . 'completed.json';

// Get action
$action = $_POST['action'] ?? null;
if (!$action) {
    echo json_encode(['success' => false, 'message' => 'Missing action']);
    exit;
}

// Handle actions
switch ($action) {
    case 'create_task':
        // Validate input
        $text = trim($_POST['text'] ?? '');
        if (empty($text) || strlen($text) > 500) {
            echo json_encode(['success' => false, 'message' => 'Invalid task text (1-500 characters)']);
            exit;
        }

        // Sanitize and validate due date/time
        $due_date = !empty($_POST['due_date']) ? $_POST['due_date'] : null;
        $due_time = !empty($_POST['due_time']) ? $_POST['due_time'] : null;

        if ($due_date && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $due_date)) {
            echo json_encode(['success' => false, 'message' => 'Invalid date format']);
            exit;
        }

        if ($due_time && !preg_match('/^\d{2}:\d{2}$/', $due_time)) {
            echo json_encode(['success' => false, 'message' => 'Invalid time format']);
            exit;
        }

        // Load existing tasks
        $tasks = [];
        if (file_exists($tasks_file)) {
            $tasks = json_decode(file_get_contents($tasks_file), true) ?? [];
        }

        // Create new task
        $task = [
            'id' => generate_task_id(),
            'text' => $text,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'due_date' => $due_date,
            'due_time' => $due_time,
            'has_due_datetime' => !empty($due_date),
            'order' => count($tasks)
        ];

        // Add to tasks
        $tasks[] = $task;

        // Save
        file_put_contents($tasks_file, json_encode($tasks, JSON_PRETTY_PRINT), LOCK_EX);

        echo json_encode(['success' => true, 'task_id' => $task['id']]);
        break;

    case 'delete_task':
        $task_id = $_POST['task_id'] ?? null;
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Missing task_id']);
            exit;
        }

        // Load tasks
        $tasks = [];
        if (file_exists($tasks_file)) {
            $tasks = json_decode(file_get_contents($tasks_file), true) ?? [];
        }

        // Filter out the task to delete
        $tasks = array_filter($tasks, function($task) use ($task_id) {
            return $task['id'] !== $task_id;
        });

        // Reindex array
        $tasks = array_values($tasks);

        // Save
        file_put_contents($tasks_file, json_encode($tasks, JSON_PRETTY_PRINT), LOCK_EX);

        echo json_encode(['success' => true]);
        break;

    case 'complete_task':
        $task_id = $_POST['task_id'] ?? null;
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Missing task_id']);
            exit;
        }

        // Load tasks
        $tasks = [];
        if (file_exists($tasks_file)) {
            $tasks = json_decode(file_get_contents($tasks_file), true) ?? [];
        }

        // Find and remove the task
        $completed_task = null;
        $tasks = array_filter($tasks, function($task) use ($task_id, &$completed_task) {
            if ($task['id'] === $task_id) {
                $completed_task = $task;
                return false;
            }
            return true;
        });

        if (!$completed_task) {
            echo json_encode(['success' => false, 'message' => 'Task not found']);
            exit;
        }

        // Add completed timestamp
        $completed_task['completed_at'] = date('Y-m-d H:i:s');

        // Load completed tasks
        $completed_tasks = [];
        if (file_exists($completed_file)) {
            $completed_tasks = json_decode(file_get_contents($completed_file), true) ?? [];
        }

        // Add to completed
        $completed_tasks[] = $completed_task;

        // Save both files
        $tasks = array_values($tasks);
        file_put_contents($tasks_file, json_encode($tasks, JSON_PRETTY_PRINT), LOCK_EX);
        file_put_contents($completed_file, json_encode($completed_tasks, JSON_PRETTY_PRINT), LOCK_EX);

        echo json_encode(['success' => true]);
        break;

    case 'restore_task':
        $task_id = $_POST['task_id'] ?? null;
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Missing task_id']);
            exit;
        }

        // Load completed tasks
        $completed_tasks = [];
        if (file_exists($completed_file)) {
            $completed_tasks = json_decode(file_get_contents($completed_file), true) ?? [];
        }

        // Find and remove the task from completed
        $restored_task = null;
        $completed_tasks = array_filter($completed_tasks, function($task) use ($task_id, &$restored_task) {
            if ($task['id'] === $task_id) {
                $restored_task = $task;
                return false;
            }
            return true;
        });

        if (!$restored_task) {
            echo json_encode(['success' => false, 'message' => 'Task not found in completion pool']);
            exit;
        }

        // Remove completed_at timestamp
        unset($restored_task['completed_at']);
        $restored_task['updated_at'] = date('Y-m-d H:i:s');

        // Load active tasks
        $tasks = [];
        if (file_exists($tasks_file)) {
            $tasks = json_decode(file_get_contents($tasks_file), true) ?? [];
        }

        // Add to active tasks
        $restored_task['order'] = count($tasks);
        $tasks[] = $restored_task;

        // Save both files
        $completed_tasks = array_values($completed_tasks);
        file_put_contents($tasks_file, json_encode($tasks, JSON_PRETTY_PRINT), LOCK_EX);
        file_put_contents($completed_file, json_encode($completed_tasks, JSON_PRETTY_PRINT), LOCK_EX);

        echo json_encode(['success' => true]);
        break;

    case 'delete_completed':
        $task_id = $_POST['task_id'] ?? null;
        if (!$task_id) {
            echo json_encode(['success' => false, 'message' => 'Missing task_id']);
            exit;
        }

        // Load completed tasks
        $completed_tasks = [];
        if (file_exists($completed_file)) {
            $completed_tasks = json_decode(file_get_contents($completed_file), true) ?? [];
        }

        // Filter out the task to delete
        $completed_tasks = array_filter($completed_tasks, function($task) use ($task_id) {
            return $task['id'] !== $task_id;
        });

        // Reindex array
        $completed_tasks = array_values($completed_tasks);

        // Save
        file_put_contents($completed_file, json_encode($completed_tasks, JSON_PRETTY_PRINT), LOCK_EX);

        echo json_encode(['success' => true]);
        break;

    case 'clear_completion_pool':
        // Load completed tasks to get count
        $completed_tasks = [];
        if (file_exists($completed_file)) {
            $completed_tasks = json_decode(file_get_contents($completed_file), true) ?? [];
        }

        $count = count($completed_tasks);

        // Clear the file
        file_put_contents($completed_file, '[]', LOCK_EX);

        echo json_encode(['success' => true, 'count' => $count]);
        break;

    default:
        echo json_encode(['success' => false, 'message' => 'Unknown action']);
        break;
}
