<?php
/**
 * TaskGram API - iCal Export
 * Exports tasks with due dates as iCalendar VTODO entries
 */

session_start();

// Helper function to find instance path
function find_instance_path($instance_id) {
    $base = __DIR__ . '/../../app/storage/plugin_instances/';

    if (!is_dir($base)) {
        return null;
    }

    // Iterate through page UUIDs
    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_taskgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }

    return null;
}

// Helper function to format datetime for iCal
function format_ical_datetime($date, $time = null) {
    if (!empty($time)) {
        // With time: YYYYMMDDTHHMMSS
        $datetime = $date . 'T' . str_replace(':', '', $time) . '00';
    } else {
        // Date only: YYYYMMDD
        $datetime = str_replace('-', '', $date);
    }
    return $datetime;
}

// Helper function to escape iCal text
function escape_ical_text($text) {
    // Escape special characters and limit line length
    $text = str_replace(["\r\n", "\n", "\r"], ' ', $text);
    $text = str_replace(['\\', ';', ','], ['\\\\', '\\;', '\\,'], $text);
    return $text;
}

// Get parameters
$instance_id = $_GET['instance_id'] ?? null;

if (!$instance_id) {
    die('Error: Missing instance_id');
}

// Sanitize instance ID
$instance_id = preg_replace('/[^a-zA-Z0-9_-]/', '', $instance_id);

// Find instance path
$instance_path = find_instance_path($instance_id);
if (!$instance_path) {
    die('Error: Instance not found');
}

$data_dir = $instance_path . '/data/';
$tasks_file = $data_dir . 'tasks.json';

if (!file_exists($tasks_file)) {
    die('Error: No tasks found');
}

// Load active tasks
$tasks = json_decode(file_get_contents($tasks_file), true) ?? [];

// Filter tasks with due dates
$tasks_with_dates = array_filter($tasks, function($task) {
    return !empty($task['due_date']);
});

if (empty($tasks_with_dates)) {
    die('Error: No tasks with due dates to export');
}

// Set headers for iCal download
header('Content-Type: text/calendar; charset=utf-8');
header('Content-Disposition: attachment; filename="taskgram-tasks-' . date('Y-m-d') . '.ics"');
header('Cache-Control: max-age=0');

// Generate iCal content
echo "BEGIN:VCALENDAR\r\n";
echo "VERSION:2.0\r\n";
echo "PRODID:-//BookGram//TaskGram Plugin//EN\r\n";
echo "CALSCALE:GREGORIAN\r\n";
echo "METHOD:PUBLISH\r\n";
echo "X-WR-CALNAME:TaskGram Tasks\r\n";
echo "X-WR-TIMEZONE:UTC\r\n";

foreach ($tasks_with_dates as $task) {
    // Create VTODO entry
    $uid = $task['id'] . '@taskgram';
    $summary = escape_ical_text($task['text']);
    $created = format_ical_datetime(date('Y-m-d', strtotime($task['created_at'])), date('H:i', strtotime($task['created_at'])));

    // Format due date/time
    $due_date = $task['due_date'];
    $due_time = $task['due_time'] ?? null;
    $due = format_ical_datetime($due_date, $due_time);

    echo "BEGIN:VTODO\r\n";
    echo "UID:" . $uid . "\r\n";
    echo "DTSTAMP:" . gmdate('Ymd\THis\Z') . "\r\n";
    echo "CREATED:" . $created . "\r\n";
    echo "SUMMARY:" . $summary . "\r\n";

    // DUE date
    if (!empty($due_time)) {
        // With time - use datetime format
        echo "DUE:" . $due . "\r\n";
    } else {
        // Date only - use VALUE=DATE
        echo "DUE;VALUE=DATE:" . $due . "\r\n";
    }

    echo "STATUS:NEEDS-ACTION\r\n";
    echo "PRIORITY:5\r\n";
    echo "SEQUENCE:0\r\n";
    echo "END:VTODO\r\n";
}

echo "END:VCALENDAR\r\n";
exit;
