<?php
/**
 * TaskGram API - CSV Export
 * Exports tasks as CSV file
 */

session_start();

// Helper function to find instance path
function find_instance_path($instance_id) {
    $base = __DIR__ . '/../../app/storage/plugin_instances/';

    if (!is_dir($base)) {
        return null;
    }

    // Iterate through page UUIDs
    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_taskgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }

    return null;
}

// Get parameters
$instance_id = $_GET['instance_id'] ?? null;
$type = $_GET['type'] ?? 'active'; // 'active' or 'completed'

if (!$instance_id) {
    die('Error: Missing instance_id');
}

// Sanitize instance ID
$instance_id = preg_replace('/[^a-zA-Z0-9_-]/', '', $instance_id);

// Sanitize type
$type = in_array($type, ['active', 'completed']) ? $type : 'active';

// Find instance path
$instance_path = find_instance_path($instance_id);
if (!$instance_path) {
    die('Error: Instance not found');
}

$data_dir = $instance_path . '/data/';

// Load appropriate file
$file = $type === 'active' ? 'tasks.json' : 'completed.json';
$file_path = $data_dir . $file;

if (!file_exists($file_path)) {
    die('Error: No tasks found');
}

$tasks = json_decode(file_get_contents($file_path), true) ?? [];

if (empty($tasks)) {
    die('Error: No tasks to export');
}

// Set headers for CSV download
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="taskgram-' . $type . '-' . date('Y-m-d') . '.csv"');
header('Cache-Control: max-age=0');

// Open output stream
$output = fopen('php://output', 'w');

// Write UTF-8 BOM for Excel compatibility
fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

// Write headers
$headers = ['Task', 'Created', 'Due Date', 'Due Time', 'Status'];
if ($type === 'completed') {
    $headers[] = 'Completed At';
}
fputcsv($output, $headers);

// Write data rows
foreach ($tasks as $task) {
    $row = [
        $task['text'],
        $task['created_at'] ?? '',
        $task['due_date'] ?? '',
        $task['due_time'] ?? '',
        $type === 'active' ? 'Active' : 'Completed'
    ];

    if ($type === 'completed') {
        $row[] = $task['completed_at'] ?? '';
    }

    fputcsv($output, $row);
}

fclose($output);
exit;
