<?php
/**
 * TaskGram Admin Panel
 * Configuration and management interface
 */

// Authenticate with BookGram
require_once dirname(dirname(__DIR__)) . '/app/core/plugin_auth.php';
$auth = bookgram_auth_check();
bookgram_require_editor();

// Get instance context
$instance_id = bookgram_get_instance_id();
$instance_path = bookgram_get_instance_path();
$username = bookgram_get_username();

if (!$instance_path) {
    die('Instance path not available');
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';
$tasks_file = $data_dir . 'tasks.json';
$completed_file = $data_dir . 'completed.json';

// Create data directory if needed
if (!is_dir($data_dir)) {
    mkdir($data_dir, 0755, true);
}

// Initialize config if not exists
if (!file_exists($config_file)) {
    $config = [
        'instance_id' => $instance_id,
        'created_at' => date('Y-m-d H:i:s'),
        'created_by' => $username,
        'title' => 'Task List',
        'description' => ''
    ];
    file_put_contents($config_file, json_encode($config, JSON_PRETTY_PRINT), LOCK_EX);

    // Create empty task files
    if (!file_exists($tasks_file)) {
        file_put_contents($tasks_file, '[]', LOCK_EX);
    }
    if (!file_exists($completed_file)) {
        file_put_contents($completed_file, '[]', LOCK_EX);
    }
}

// Load config
$config = json_decode(file_get_contents($config_file), true) ?? [];

// Load tasks for statistics
$tasks = [];
if (file_exists($tasks_file)) {
    $tasks = json_decode(file_get_contents($tasks_file), true) ?? [];
}

$completed_tasks = [];
if (file_exists($completed_file)) {
    $completed_tasks = json_decode(file_get_contents($completed_file), true) ?? [];
}

// Handle form submission
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'update_config') {
        $config['title'] = trim($_POST['title'] ?? 'Task List');
        $config['description'] = trim($_POST['description'] ?? '');
        $config['updated_at'] = date('Y-m-d H:i:s');
        $config['updated_by'] = $username;

        file_put_contents($config_file, json_encode($config, JSON_PRETTY_PRINT), LOCK_EX);

        $message = 'Configuration updated successfully!';
        $message_type = 'success';

        // Signal parent to reload
        echo '<script>window.parent.postMessage("plugin-admin-saved", "*");</script>';

    } elseif ($action === 'clear_completed') {
        $count = count($completed_tasks);
        file_put_contents($completed_file, '[]', LOCK_EX);
        $completed_tasks = [];

        $message = "Cleared $count completed task(s)!";
        $message_type = 'success';
    }
}

// Calculate statistics
$total_active = count($tasks);
$total_completed = count($completed_tasks);
$tasks_with_dates = array_filter($tasks, function($task) {
    return !empty($task['due_date']);
});
$total_with_dates = count($tasks_with_dates);

// Count overdue tasks
$overdue_count = 0;
$now = new DateTime();
foreach ($tasks_with_dates as $task) {
    $due = new DateTime($task['due_date']);
    if (!empty($task['due_time'])) {
        $time_parts = explode(':', $task['due_time']);
        $due->setTime((int)$time_parts[0], (int)$time_parts[1]);
    } else {
        $due->setTime(23, 59, 59);
    }
    if ($due < $now) {
        $overdue_count++;
    }
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>TaskGram Configuration</title>
    <link rel="stylesheet" href="/public/css/os2-theme.css">
    <style>
        body {
            background: #008080;
            padding: 20px;
            font-family: "MS Sans Serif", Tahoma, sans-serif;
            margin: 0;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
        }
        .window {
            background: #c0c0c0;
            border: 2px solid #fff;
            box-shadow: 2px 2px 0 #000;
            margin-bottom: 20px;
        }
        .window-title-bar {
            background: linear-gradient(to right, #000080, #1084d0);
            color: #fff;
            padding: 3px 5px;
            font-weight: bold;
            font-size: 14px;
        }
        .window-content {
            padding: 20px;
        }
        h2 {
            margin-top: 0;
            color: #000080;
            font-size: 16px;
        }
        label {
            display: block;
            margin-top: 15px;
            margin-bottom: 5px;
            font-weight: bold;
            font-size: 13px;
        }
        input[type="text"], textarea {
            width: 100%;
            padding: 6px 8px;
            border: 2px inset #808080;
            font-family: inherit;
            font-size: 13px;
            box-sizing: border-box;
        }
        textarea {
            resize: vertical;
        }
        button {
            padding: 6px 15px;
            background: #c0c0c0;
            border: 2px outset #808080;
            cursor: pointer;
            font-family: inherit;
            font-size: 12px;
            margin-top: 10px;
            margin-right: 5px;
        }
        button:active {
            border-style: inset;
        }
        .btn-primary {
            background: #000080;
            color: #ffffff;
        }
        .btn-danger {
            background: #cc0000;
            color: #ffffff;
        }
        .message {
            padding: 10px;
            margin-bottom: 15px;
            border: 2px solid;
        }
        .message.success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        .message.error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        .stats {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 10px;
            margin-top: 15px;
        }
        .stat-box {
            background: #fff;
            border: 2px inset #808080;
            padding: 15px;
            text-align: center;
        }
        .stat-number {
            font-size: 32px;
            font-weight: bold;
            color: #000080;
        }
        .stat-label {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
        .info-text {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Configuration Window -->
        <div class="window">
            <div class="window-title-bar">
                <span class="title">TaskGram Configuration</span>
            </div>
            <div class="window-content">
                <?php if ($message): ?>
                    <div class="message <?php echo $message_type; ?>">
                        <?php echo htmlspecialchars($message); ?>
                    </div>
                <?php endif; ?>

                <h2>Settings</h2>
                <form method="POST">
                    <input type="hidden" name="action" value="update_config">

                    <label>Title:</label>
                    <input type="text" name="title" value="<?php echo htmlspecialchars($config['title'] ?? 'Task List'); ?>" required>
                    <div class="info-text">The title displayed at the top of the task list</div>

                    <label>Description (optional):</label>
                    <textarea name="description" rows="3"><?php echo htmlspecialchars($config['description'] ?? ''); ?></textarea>
                    <div class="info-text">A brief description shown below the title</div>

                    <button type="submit" class="btn-primary">Save Changes</button>
                </form>
            </div>
        </div>

        <!-- Statistics Window -->
        <div class="window">
            <div class="window-title-bar">
                <span class="title">Statistics</span>
            </div>
            <div class="window-content">
                <h2>Task Overview</h2>
                <div class="stats">
                    <div class="stat-box">
                        <div class="stat-number"><?php echo $total_active; ?></div>
                        <div class="stat-label">Active Tasks</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-number"><?php echo $total_completed; ?></div>
                        <div class="stat-label">Completed Tasks</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-number"><?php echo $total_with_dates; ?></div>
                        <div class="stat-label">With Due Dates</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-number" style="<?php echo $overdue_count > 0 ? 'color: #cc0000;' : ''; ?>">
                            <?php echo $overdue_count; ?>
                        </div>
                        <div class="stat-label">Overdue</div>
                    </div>
                </div>

                <div style="margin-top: 20px; padding-top: 20px; border-top: 2px solid #808080;">
                    <p style="margin: 0 0 10px 0; font-size: 13px;"><strong>Instance ID:</strong> <?php echo htmlspecialchars($instance_id); ?></p>
                    <p style="margin: 0; font-size: 13px;"><strong>Created:</strong> <?php echo htmlspecialchars($config['created_at'] ?? 'Unknown'); ?> by <?php echo htmlspecialchars($config['created_by'] ?? 'Unknown'); ?></p>
                </div>
            </div>
        </div>

        <!-- Bulk Actions Window -->
        <div class="window">
            <div class="window-title-bar">
                <span class="title">Bulk Actions</span>
            </div>
            <div class="window-content">
                <h2>Manage Completed Tasks</h2>
                <p style="font-size: 13px;">Clear all completed tasks from the completion pool. This action cannot be undone.</p>

                <form method="POST" onsubmit="return confirm('Are you sure you want to delete all completed tasks? This cannot be undone.');">
                    <input type="hidden" name="action" value="clear_completed">
                    <button type="submit" class="btn-danger" <?php echo $total_completed === 0 ? 'disabled' : ''; ?>>
                        Clear All Completed Tasks (<?php echo $total_completed; ?>)
                    </button>
                </form>
            </div>
        </div>

        <!-- Help Window -->
        <div class="window">
            <div class="window-title-bar">
                <span class="title">Usage Help</span>
            </div>
            <div class="window-content">
                <h2>How to Use TaskGram</h2>
                <ul style="font-size: 13px; line-height: 1.6;">
                    <li><strong>Add Tasks:</strong> Use the form at the top to add new tasks. Optionally set due dates and times.</li>
                    <li><strong>Complete Tasks:</strong> Check the checkbox next to a task to mark it as complete. It moves to the completion pool.</li>
                    <li><strong>Delete Tasks:</strong> Click the "Delete" button to permanently remove a task.</li>
                    <li><strong>Search:</strong> Use the search bar to filter tasks by text.</li>
                    <li><strong>Export:</strong> Export active or completed tasks as CSV, or export tasks with due dates as iCal.</li>
                    <li><strong>Restore:</strong> From the completion pool, you can restore tasks back to active or delete them permanently.</li>
                </ul>

                <h2 style="margin-top: 20px;">Embedding in Pages</h2>
                <p style="font-size: 13px;">Use this shortcode to embed this task list in any page:</p>
                <pre style="background: #fff; padding: 10px; border: 2px inset #808080; font-family: monospace; font-size: 12px;">[plugin:taskgram id=<?php echo htmlspecialchars($instance_id); ?>]</pre>
            </div>
        </div>
    </div>
</body>
</html>
