<?php
/**
 * SocialGram Embed Handler
 * Renders discussion interface inline in article
 */

// Get instance context
$instance_path = $_ENV['BOOKGRAM_INSTANCE_PATH'] ?? null;
$instance_id = $_ENV['BOOKGRAM_INSTANCE_ID'] ?? null;
$page_uuid = $_ENV['BOOKGRAM_PAGE_UUID'] ?? null;

if (!$instance_path || !$instance_id) {
    echo '<p style="color: #cc0000;">Error: SocialGram instance context not available.</p>';
    return;
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';

// Check if configured
if (!file_exists($config_file)) {
    echo '<div style="padding: 20px; background: #fff3cd; border: 2px solid #ff6600; margin: 20px 0;">';
    echo '<h3>💬 Discussion Group Not Configured</h3>';
    echo '<p><a href="#" onclick="event.preventDefault(); openPluginAdmin(\'' . htmlspecialchars($page_uuid) . '\', \'plugin_socialgram\', \'' . htmlspecialchars($instance_id) . '\', \'SocialGram\');" class="config-link">Setup Discussion Group</a></p>';
    echo '</div>';
    return;
}

// Load config
$config = json_decode(file_get_contents($config_file), true);

// Ensure config has defaults
if (!isset($config['allow_anonymous'])) {
    $config['allow_anonymous'] = false;
}
if (!isset($config['group_name'])) {
    $config['group_name'] = 'Discussion Group';
}
if (!isset($config['description'])) {
    $config['description'] = '';
}

// Check if user is logged in (for auth check)
$is_logged_in = false;
$current_user = 'Anonymous';
$is_admin = false;

// Load BookGram auth functions if available
if (file_exists(__DIR__ . '/../app/core/functions.php')) {
    require_once __DIR__ . '/../app/core/functions.php';
}

if (function_exists('is_logged_in')) {
    $is_logged_in = is_logged_in();
    if ($is_logged_in) {
        $current_user = get_current_username() ?? 'User';
        $is_admin = is_admin();
    }
}

// Load posts
$posts_dir = $data_dir . 'posts/';
$posts = [];

if (is_dir($posts_dir)) {
    $files = glob($posts_dir . '*.json');
    foreach ($files as $file) {
        $post = json_decode(file_get_contents($file), true);
        if ($post) {
            $posts[] = $post;
        }
    }
}

// Sort by timestamp (newest first)
usort($posts, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// Helper function to parse URLs and make them clickable
function parse_links($text) {
    $text = htmlspecialchars($text);
    $pattern = '/(https?:\/\/[^\s]+)/i';
    return preg_replace($pattern, '<a href="$1" target="_blank" rel="noopener">$1</a>', $text);
}

// Helper function to detect and embed media
function embed_media($url) {
    // YouTube
    if (preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]+)/', $url, $matches)) {
        return '<div class="sg-media-embed"><iframe width="560" height="315" src="https://www.youtube.com/embed/' . $matches[1] . '" frameborder="0" allowfullscreen></iframe></div>';
    }
    // Vimeo
    if (preg_match('/vimeo\.com\/(\d+)/', $url, $matches)) {
        return '<div class="sg-media-embed"><iframe src="https://player.vimeo.com/video/' . $matches[1] . '" width="560" height="315" frameborder="0" allowfullscreen></iframe></div>';
    }
    return '';
}

// Recursive function to render posts and replies
function render_post($post, $all_posts, $level = 0, $is_admin = false) {
    $post_id = $post['id'];
    $level_class = $level > 0 ? 'level-' . min($level, 4) : '';

    echo '<div class="sg-post ' . $level_class . '" data-post-id="' . htmlspecialchars($post_id) . '">';

    // Header
    echo '<div class="sg-post-header">';
    echo '<div class="sg-post-meta">';
    echo '<span class="sg-post-author">' . htmlspecialchars($post['author']) . '</span>';
    if (!empty($post['is_admin'])) {
        echo '<span class="sg-admin-badge">ADMIN</span>';
    }
    echo ' • ' . date('M j, Y g:i A', strtotime($post['created_at']));
    echo '</div>';
    echo '</div>';

    // Content
    echo '<div class="sg-post-content">';

    // Parse and display text with clickable links
    $content = parse_links($post['content']);
    echo nl2br($content);

    // Image if present
    if (!empty($post['image'])) {
        echo '<div class="sg-post-image">';
        echo '<img src="' . htmlspecialchars($post['image']) . '" alt="Post image">';
        echo '</div>';
    }

    // Media embed if URL detected
    if (!empty($post['media_url'])) {
        echo embed_media($post['media_url']);
    }

    // Edited indicator
    if (!empty($post['edited_at'])) {
        echo '<div class="sg-edited">Edited ' . date('M j, Y g:i A', strtotime($post['edited_at'])) . '</div>';
    }

    echo '</div>';

    // Actions
    echo '<div class="sg-post-actions">';

    // Reply button (only if not at max depth)
    if ($level < 4) {
        echo '<button onclick="toggleReply(\'' . $post_id . '\')">Reply</button>';
    }

    // Admin actions
    if ($is_admin) {
        echo '<button onclick="editPost(\'' . $post_id . '\')">Edit</button>';
        echo '<button onclick="deletePost(\'' . $post_id . '\')">Delete</button>';
    }

    echo '</div>';

    // Reply form (hidden by default)
    echo '<div class="sg-reply-form" id="reply-form-' . $post_id . '">';
    echo '<form onsubmit="submitReply(event, \'' . $post_id . '\')">';
    echo '<textarea name="content" placeholder="Write your reply..." required></textarea><br>';
    echo '<button type="submit">Post Reply</button> ';
    echo '<button type="button" onclick="toggleReply(\'' . $post_id . '\')">Cancel</button>';
    echo '</form>';
    echo '</div>';

    // Render replies
    $replies = array_filter($all_posts, function($p) use ($post_id) {
        return ($p['parent_id'] ?? null) === $post_id;
    });

    if (!empty($replies)) {
        foreach ($replies as $reply) {
            render_post($reply, $all_posts, $level + 1, $is_admin);
        }
    }

    echo '</div>';
}
?>

<div class="socialgram-container" id="socialgram-<?php echo htmlspecialchars($instance_id); ?>">
    <div class="sg-header">
        <h3>💬 <?php echo htmlspecialchars($config['group_name']); ?></h3>
        <?php if (!empty($config['description'])): ?>
            <p><?php echo htmlspecialchars($config['description']); ?></p>
        <?php endif; ?>
    </div>

    <!-- Search Box -->
    <div class="sg-search">
        <input type="text" id="sg-search-input" placeholder="🔍 Search discussions..." onkeyup="searchPosts()">
    </div>

    <!-- Post Form -->
    <?php if ($config['allow_anonymous'] || $is_logged_in): ?>
        <div class="sg-post-form">
            <h4>Start a Discussion</h4>
            <form id="sg-new-post-form" enctype="multipart/form-data">
                <?php if ($config['allow_anonymous'] && !$is_logged_in): ?>
                    <input type="text" name="author" placeholder="Your name (optional)" value="Anonymous">
                <?php endif; ?>
                <textarea name="content" placeholder="What's on your mind?" required></textarea>
                <div style="margin: 10px 0;">
                    <label>📷 Image (max 2MB): <input type="file" name="image" accept="image/*"></label>
                </div>
                <div style="margin: 10px 0;">
                    <input type="text" name="media_url" placeholder="🎥 Media URL (YouTube, Vimeo)" style="width: 300px;">
                </div>
                <button type="submit">Post</button>
            </form>
        </div>
    <?php else: ?>
        <div style="padding: 15px; background: #f0f0f0; border: 1px solid #808080; margin-bottom: 20px;">
            <p><strong>Login required to post.</strong> <a href="/login">Login to BookGram</a></p>
        </div>
    <?php endif; ?>

    <!-- Posts List -->
    <div class="sg-posts" id="sg-posts-container">
        <?php if (empty($posts)): ?>
            <div class="sg-no-posts">No posts yet. Be the first to start a discussion!</div>
        <?php else: ?>
            <?php
            // Render top-level posts only (replies are rendered recursively)
            $top_level_posts = array_filter($posts, function($p) {
                return empty($p['parent_id']);
            });
            foreach ($top_level_posts as $post) {
                render_post($post, $posts, 0, $is_admin);
            }
            ?>
        <?php endif; ?>
    </div>
</div>

<script>
// Configuration
const instanceId = '<?php echo htmlspecialchars($instance_id); ?>';
const instancePath = '/plugin_socialgram/api/';
const isAdmin = <?php echo $is_admin ? 'true' : 'false'; ?>;
const currentUser = '<?php echo addslashes($current_user); ?>';

// Submit new post
document.getElementById('sg-new-post-form')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    formData.append('instance_id', instanceId);
    formData.append('action', 'create_post');

    <?php if ($is_logged_in): ?>
    formData.append('author', currentUser);
    formData.append('is_admin', isAdmin ? '1' : '0');
    <?php endif; ?>

    try {
        const response = await fetch(instancePath + 'post.php', {
            method: 'POST',
            body: formData
        });
        const result = await response.json();
        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + result.message);
        }
    } catch (error) {
        alert('Error posting. Please try again.');
    }
});

// Submit reply
async function submitReply(event, parentId) {
    event.preventDefault();
    const form = event.target;
    const formData = new FormData(form);
    formData.append('instance_id', instanceId);
    formData.append('action', 'create_post');
    formData.append('parent_id', parentId);
    formData.append('author', currentUser);
    formData.append('is_admin', isAdmin ? '1' : '0');

    try {
        const response = await fetch(instancePath + 'post.php', {
            method: 'POST',
            body: formData
        });
        const result = await response.json();
        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + result.message);
        }
    } catch (error) {
        alert('Error posting reply. Please try again.');
    }
}

// Toggle reply form
function toggleReply(postId) {
    const form = document.getElementById('reply-form-' + postId);
    form.classList.toggle('active');
}

// Delete post (admin only)
async function deletePost(postId) {
    if (!confirm('Delete this post and all replies?')) return;

    try {
        const formData = new FormData();
        formData.append('instance_id', instanceId);
        formData.append('action', 'delete_post');
        formData.append('post_id', postId);

        const response = await fetch(instancePath + 'post.php', {
            method: 'POST',
            body: formData
        });
        const result = await response.json();
        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + result.message);
        }
    } catch (error) {
        alert('Error deleting post. Please try again.');
    }
}

// Edit post (admin only)
function editPost(postId) {
    const post = document.querySelector('[data-post-id="' + postId + '"]');
    const content = post.querySelector('.sg-post-content').textContent.trim();
    const newContent = prompt('Edit post:', content);

    if (newContent !== null && newContent.trim() !== '') {
        updatePost(postId, newContent);
    }
}

async function updatePost(postId, content) {
    try {
        const formData = new FormData();
        formData.append('instance_id', instanceId);
        formData.append('action', 'edit_post');
        formData.append('post_id', postId);
        formData.append('content', content);

        const response = await fetch(instancePath + 'post.php', {
            method: 'POST',
            body: formData
        });
        const result = await response.json();
        if (result.success) {
            location.reload();
        } else {
            alert('Error: ' + result.message);
        }
    } catch (error) {
        alert('Error updating post. Please try again.');
    }
}

// Search posts
function searchPosts() {
    const searchTerm = document.getElementById('sg-search-input').value.toLowerCase();
    const posts = document.querySelectorAll('.sg-post');

    posts.forEach(post => {
        const content = post.textContent.toLowerCase();
        if (content.includes(searchTerm)) {
            post.style.display = '';
        } else {
            post.style.display = 'none';
        }
    });
}
</script>
