<?php
/**
 * SocialGram Post API
 * Handles creating, editing, and deleting posts
 */

session_start();
header('Content-Type: application/json');

// Get instance from request
$instance_id = $_POST['instance_id'] ?? null;
$action = $_POST['action'] ?? null;

if (!$instance_id || !$action) {
    echo json_encode(['success' => false, 'message' => 'Missing required parameters']);
    exit;
}

// Build instance path
$instance_base = __DIR__ . '/../../app/storage/plugin_instances/';

// Find the instance (we need to search for it since we don't have page_uuid easily)
function find_instance_path($instance_id) {
    $base = __DIR__ . '/../../app/storage/plugin_instances/';

    if (!is_dir($base)) {
        return null;
    }

    // Iterate through page UUIDs
    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_socialgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }

    return null;
}

$instance_path = find_instance_path($instance_id);

if (!$instance_path) {
    echo json_encode(['success' => false, 'message' => 'Instance not found']);
    exit;
}

$data_dir = $instance_path . '/data/';
$posts_dir = $data_dir . 'posts/';
$uploads_dir = $data_dir . 'uploads/';
$config_file = $data_dir . 'config.json';

// Ensure directories exist
if (!is_dir($posts_dir)) @mkdir($posts_dir, 0755, true);
if (!is_dir($uploads_dir)) @mkdir($uploads_dir, 0755, true);

// Load config
$config = file_exists($config_file) ? json_decode(file_get_contents($config_file), true) : [];

// Helper: Resize image to max 2MP
function resize_image($source_path, $dest_path, $max_pixels = 2000000) {
    // Get image info
    $image_info = getimagesize($source_path);
    if (!$image_info) return false;

    list($width, $height, $type) = $image_info;
    $current_pixels = $width * $height;

    // Load image based on type
    switch ($type) {
        case IMAGETYPE_JPEG:
            $source = imagecreatefromjpeg($source_path);
            break;
        case IMAGETYPE_PNG:
            $source = imagecreatefrompng($source_path);
            break;
        case IMAGETYPE_GIF:
            $source = imagecreatefromgif($source_path);
            break;
        default:
            return false;
    }

    if (!$source) return false;

    // Calculate new dimensions if needed
    if ($current_pixels > $max_pixels) {
        $scale = sqrt($max_pixels / $current_pixels);
        $new_width = (int)($width * $scale);
        $new_height = (int)($height * $scale);
    } else {
        $new_width = $width;
        $new_height = $height;
    }

    // Resize
    $dest_image = imagecreatetruecolor($new_width, $new_height);

    // Preserve transparency for PNG
    if ($type == IMAGETYPE_PNG) {
        imagealphablending($dest_image, false);
        imagesavealpha($dest_image, true);
    }

    imagecopyresampled($dest_image, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);

    // Save
    $result = imagejpeg($dest_image, $dest_path, 85);

    imagedestroy($source);
    imagedestroy($dest_image);

    return $result;
}

// Action: Create Post
if ($action === 'create_post') {
    $author = trim($_POST['author'] ?? 'Anonymous');
    $content = trim($_POST['content'] ?? '');
    $parent_id = $_POST['parent_id'] ?? null;
    $media_url = trim($_POST['media_url'] ?? '');
    $is_admin = !empty($_POST['is_admin']);

    if (empty($content)) {
        echo json_encode(['success' => false, 'message' => 'Content is required']);
        exit;
    }

    // Generate unique post ID
    $post_id = uniqid('post_', true);

    // Handle image upload
    $image_path = null;
    if (!empty($_FILES['image']['tmp_name'])) {
        $upload_file = $uploads_dir . $post_id . '.jpg';
        $relative_path = '/plugin_socialgram/api/../../../app/storage/plugin_instances/' .
                        basename(dirname(dirname($instance_path))) . '/plugin_socialgram/' .
                        $instance_id . '/data/uploads/' . $post_id . '.jpg';

        if (resize_image($_FILES['image']['tmp_name'], $upload_file)) {
            $image_path = $relative_path;
        }
    }

    // Create post data
    $post = [
        'id' => $post_id,
        'author' => $author,
        'content' => $content,
        'parent_id' => $parent_id,
        'image' => $image_path,
        'media_url' => $media_url,
        'is_admin' => $is_admin,
        'created_at' => date('Y-m-d H:i:s'),
        'edited_at' => null
    ];

    // Save post
    $post_file = $posts_dir . $post_id . '.json';
    if (file_put_contents($post_file, json_encode($post, JSON_PRETTY_PRINT))) {
        echo json_encode(['success' => true, 'post_id' => $post_id]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to save post']);
    }
    exit;
}

// Action: Edit Post (Admin only)
if ($action === 'edit_post') {
    $post_id = $_POST['post_id'] ?? null;
    $new_content = trim($_POST['content'] ?? '');

    if (!$post_id || empty($new_content)) {
        echo json_encode(['success' => false, 'message' => 'Missing parameters']);
        exit;
    }

    $post_file = $posts_dir . $post_id . '.json';

    if (!file_exists($post_file)) {
        echo json_encode(['success' => false, 'message' => 'Post not found']);
        exit;
    }

    $post = json_decode(file_get_contents($post_file), true);
    $post['content'] = $new_content;
    $post['edited_at'] = date('Y-m-d H:i:s');

    if (file_put_contents($post_file, json_encode($post, JSON_PRETTY_PRINT))) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to save changes']);
    }
    exit;
}

// Action: Delete Post (Admin only)
if ($action === 'delete_post') {
    $post_id = $_POST['post_id'] ?? null;

    if (!$post_id) {
        echo json_encode(['success' => false, 'message' => 'Missing post ID']);
        exit;
    }

    // Delete post file
    $post_file = $posts_dir . $post_id . '.json';
    $deleted = false;

    if (file_exists($post_file)) {
        $post = json_decode(file_get_contents($post_file), true);

        // Delete associated image if exists
        if (!empty($post['image'])) {
            $image_file = $uploads_dir . $post_id . '.jpg';
            if (file_exists($image_file)) {
                @unlink($image_file);
            }
        }

        $deleted = unlink($post_file);
    }

    // Also delete all replies to this post
    $all_posts = glob($posts_dir . '*.json');
    foreach ($all_posts as $file) {
        $p = json_decode(file_get_contents($file), true);
        if (isset($p['parent_id']) && $p['parent_id'] === $post_id) {
            unlink($file);
            // Delete reply image if exists
            if (!empty($p['image'])) {
                $reply_image = $uploads_dir . $p['id'] . '.jpg';
                if (file_exists($reply_image)) {
                    @unlink($reply_image);
                }
            }
        }
    }

    echo json_encode(['success' => $deleted]);
    exit;
}

echo json_encode(['success' => false, 'message' => 'Invalid action']);
