<?php
/**
 * SocialGram Admin Panel
 * Moderate discussion content
 */

// Use BookGram authentication
require_once dirname(dirname(__DIR__)) . '/app/core/plugin_auth.php';
$auth = bookgram_auth_check();
bookgram_require_editor();

// Get instance context
$instance_id = bookgram_get_instance_id();
$instance_path = bookgram_get_instance_path();

// Validate instance path
if (!$instance_path || !is_dir($instance_path)) {
    die('<html><body><h1>Error</h1><p>Instance path not found.</p></body></html>');
}

// Build URL parameters
$token = $_GET['bga_token'] ?? '';
$instance_param = $_GET['instance'] ?? '';
$url_params = $token ? "?bga_token=" . urlencode($token) . "&instance=" . urlencode($instance_param) : '';

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';
$posts_dir = $data_dir . 'posts/';

// Load config
$config = file_exists($config_file) ? json_decode(file_get_contents($config_file), true) : [];

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_settings') {
    $config['group_name'] = trim($_POST['group_name'] ?? $config['group_name'] ?? 'Discussion Group');
    $config['description'] = trim($_POST['description'] ?? $config['description'] ?? '');
    $config['allow_anonymous'] = isset($_POST['allow_anonymous']);

    if (file_put_contents($config_file, json_encode($config, JSON_PRETTY_PRINT))) {
        $success_message = 'Settings updated successfully!';
    } else {
        $error_message = 'Failed to save settings.';
    }
}

// Load all posts
$posts = [];
if (is_dir($posts_dir)) {
    $files = glob($posts_dir . '*.json');
    foreach ($files as $file) {
        $post = json_decode(file_get_contents($file), true);
        if ($post) {
            $posts[] = $post;
        }
    }
}

// Sort by newest first
usort($posts, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

$total_posts = count($posts);
$top_level_posts = array_filter($posts, function($p) { return empty($p['parent_id']); });
$replies = array_filter($posts, function($p) { return !empty($p['parent_id']); });
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SocialGram Admin - <?php echo htmlspecialchars($instance_id); ?></title>
    <link rel="stylesheet" href="/public/css/os2-theme.css">
    <style>
        body { background: #008080; padding: 20px; }
        .admin-container { max-width: 1200px; margin: 0 auto; }
        .stats { display: flex; gap: 20px; margin-bottom: 20px; }
        .stat-box { background: #ffffff; padding: 15px; border: 2px solid #000080; flex: 1; }
        .stat-box h3 { margin: 0 0 10px 0; color: #000080; }
        .stat-number { font-size: 2em; font-weight: bold; color: #000080; }
        .post-item { background: #f9f9f9; padding: 15px; margin-bottom: 10px; border: 1px solid #808080; }
        .post-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; }
        .post-meta { color: #666; font-size: 0.9em; }
        .post-actions button { padding: 5px 10px; margin-right: 5px; background: #c0c0c0; border: 1px outset #808080; cursor: pointer; }
        .post-actions button.delete { background: #ff4444; color: white; }
        .info-box { background: #e7f3ff; padding: 15px; border-left: 4px solid #007bff; margin-bottom: 20px; }
    </style>
</head>
<body>
    <div class="admin-container">
        <div class="window">
            <div class="window-title-bar">
                <span class="title">SocialGram Admin - <?php echo htmlspecialchars($config['group_name'] ?? 'Discussion Group'); ?></span>
            </div>
            <div class="window-content">
                <div class="info-box">
                    <strong>Instance:</strong> <?php echo htmlspecialchars($instance_id); ?><br>
                    <strong>Manager:</strong> <?php echo htmlspecialchars(bookgram_get_username()); ?> (<?php echo bookgram_is_admin() ? 'Admin' : 'Editor'; ?>)
                </div>

                <?php if (isset($success_message)): ?>
                    <div style="background: #ccffcc; padding: 10px; border: 2px solid #00cc00; margin-bottom: 20px;">
                        ✓ <?php echo htmlspecialchars($success_message); ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($error_message)): ?>
                    <div style="background: #ffcccc; padding: 10px; border: 2px solid #cc0000; margin-bottom: 20px;">
                        ✗ <?php echo htmlspecialchars($error_message); ?>
                    </div>
                <?php endif; ?>

                <!-- Settings Section -->
                <h2>Settings</h2>
                <div style="background: #f0f0f0; padding: 15px; border: 2px inset #c0c0c0; margin-bottom: 20px;">
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="update_settings">

                        <div style="margin-bottom: 15px;">
                            <label><strong>Group Name:</strong></label><br>
                            <input type="text" name="group_name" value="<?php echo htmlspecialchars($config['group_name'] ?? 'Discussion Group'); ?>" required
                                   style="width: 400px; padding: 6px; border: 2px inset #c0c0c0;">
                        </div>

                        <div style="margin-bottom: 15px;">
                            <label><strong>Description:</strong></label><br>
                            <textarea name="description" rows="3"
                                      style="width: 400px; padding: 6px; border: 2px inset #c0c0c0;"><?php echo htmlspecialchars($config['description'] ?? ''); ?></textarea>
                        </div>

                        <div style="margin-bottom: 15px;">
                            <label>
                                <input type="checkbox" name="allow_anonymous" value="1" <?php echo ($config['allow_anonymous'] ?? false) ? 'checked' : ''; ?>>
                                <strong>Allow anonymous posting</strong> (users don't need to login)
                            </label>
                        </div>

                        <button type="submit" style="padding: 8px 20px; background: #000080; color: #fff; border: 2px outset #c0c0c0; cursor: pointer;">
                            Save Settings
                        </button>
                    </form>
                </div>

                <h2>Statistics</h2>
                <div class="stats">
                    <div class="stat-box">
                        <h3>Total Posts</h3>
                        <div class="stat-number"><?php echo $total_posts; ?></div>
                    </div>
                    <div class="stat-box">
                        <h3>Top-Level Posts</h3>
                        <div class="stat-number"><?php echo count($top_level_posts); ?></div>
                    </div>
                    <div class="stat-box">
                        <h3>Replies</h3>
                        <div class="stat-number"><?php echo count($replies); ?></div>
                    </div>
                </div>

                <h2>Recent Posts</h2>
                <?php if (empty($posts)): ?>
                    <p><em>No posts yet.</em></p>
                <?php else: ?>
                    <?php foreach (array_slice($posts, 0, 20) as $post): ?>
                        <div class="post-item">
                            <div class="post-header">
                                <div class="post-meta">
                                    <strong><?php echo htmlspecialchars($post['author']); ?></strong>
                                    <?php if (!empty($post['is_admin'])): ?>
                                        <span style="background: #cc0000; color: white; padding: 2px 5px; font-size: 0.8em; border-radius: 3px;">ADMIN</span>
                                    <?php endif; ?>
                                    • <?php echo date('M j, Y g:i A', strtotime($post['created_at'])); ?>
                                    <?php if (!empty($post['parent_id'])): ?>
                                        <span style="color: #0000ff;">↳ Reply</span>
                                    <?php endif; ?>
                                </div>
                                <div class="post-actions">
                                    <button onclick="viewPost('<?php echo htmlspecialchars($post['id']); ?>')">View</button>
                                    <button onclick="editPost('<?php echo htmlspecialchars($post['id']); ?>')">Edit</button>
                                    <button class="delete" onclick="deletePost('<?php echo htmlspecialchars($post['id']); ?>')">Delete</button>
                                </div>
                            </div>
                            <div class="post-content">
                                <?php echo nl2br(htmlspecialchars(substr($post['content'], 0, 200))); ?>
                                <?php if (strlen($post['content']) > 200): ?>...<?php endif; ?>
                            </div>
                            <?php if (!empty($post['image'])): ?>
                                <div style="margin-top: 10px;">
                                    📷 <em>Contains image</em>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

                <p style="margin-top: 20px;">
                    <a href="/">← Return to BookGram</a>
                </p>
            </div>
        </div>
    </div>

    <script>
    const instanceId = '<?php echo htmlspecialchars($instance_id); ?>';

    function viewPost(postId) {
        // Could implement a modal or navigate to discussion
        alert('Post ID: ' + postId);
    }

    async function editPost(postId) {
        const newContent = prompt('Edit post content:');
        if (!newContent) return;

        try {
            const formData = new FormData();
            formData.append('instance_id', instanceId);
            formData.append('action', 'edit_post');
            formData.append('post_id', postId);
            formData.append('content', newContent);

            const response = await fetch('/plugin_socialgram/api/post.php', {
                method: 'POST',
                body: formData
            });
            const result = await response.json();
            if (result.success) {
                location.reload();
            } else {
                alert('Error: ' + result.message);
            }
        } catch (error) {
            alert('Error editing post.');
        }
    }

    async function deletePost(postId) {
        if (!confirm('Delete this post and all its replies? This cannot be undone.')) return;

        try {
            const formData = new FormData();
            formData.append('instance_id', instanceId);
            formData.append('action', 'delete_post');
            formData.append('post_id', postId);

            const response = await fetch('/plugin_socialgram/api/post.php', {
                method: 'POST',
                body: formData
            });
            const result = await response.json();
            if (result.success) {
                location.reload();
            } else {
                alert('Error: ' + result.message);
            }
        } catch (error) {
            alert('Error deleting post.');
        }
    }

    // Modal mode detection and parent notification
    (function() {
        const urlParams = new URLSearchParams(window.location.search);
        const isModal = urlParams.get('modal') === '1';

        if (isModal) {
            // Hide "Return to BookGram" link in modal mode
            const returnLink = document.querySelector('a[href="/"]');
            if (returnLink && returnLink.textContent.includes('Return to BookGram')) {
                returnLink.style.display = 'none';
            }

            // Notify parent on successful save
            <?php if (isset($success_message) && $success_message): ?>
            setTimeout(function() {
                if (window.parent && window.parent !== window) {
                    window.parent.postMessage('plugin-admin-saved', '*');
                }
            }, 1500);
            <?php endif; ?>

            // Add modal styling adjustments
            document.body.style.margin = '0';
            document.body.style.padding = '10px';
            document.body.style.background = '#008080';
        }
    })();
    </script>
</body>
</html>
