<?php
// Common authentication and setup
require_once '_auth_header.php';

$survey_file = $data_dir . 'survey.json';
$change_log_file = $data_dir . 'change_log.json';

// Handle Save Request (sent via JavaScript Fetch)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = file_get_contents('php://input');
    $decoded_data = json_decode($data, true);

    if (json_last_error() === JSON_ERROR_NONE) {
        // Load existing survey for comparison
        $old_survey = file_exists($survey_file) ? json_decode(file_get_contents($survey_file), true) : [];

        // Save new survey
        file_put_contents($survey_file, json_encode($decoded_data, JSON_PRETTY_PRINT), LOCK_EX);

        // Log the change
        $change_log = file_exists($change_log_file) ? json_decode(file_get_contents($change_log_file), true) : [];
        if (!is_array($change_log)) {
            $change_log = [];
        }

        $change_entry = [
            'change_id' => 'chg_' . bin2hex(random_bytes(8)),
            'timestamp' => date('Y-m-d H:i:s'),
            'user' => $_SESSION['bookgram_user'] ?? 'unknown',
            'old_field_count' => count($old_survey),
            'new_field_count' => count($decoded_data),
            'changes_summary' => generate_changes_summary($old_survey, $decoded_data)
        ];

        $change_log[] = $change_entry;

        // Keep only last 100 changes
        if (count($change_log) > 100) {
            $change_log = array_slice($change_log, -100);
        }

        file_put_contents($change_log_file, json_encode($change_log, JSON_PRETTY_PRINT), LOCK_EX);

        echo json_encode(['status' => 'success', 'message' => 'Survey saved!']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Invalid data.']);
    }
    exit;
}

// Helper function to generate change summary
function generate_changes_summary($old, $new) {
    $old_count = count($old);
    $new_count = count($new);

    if ($old_count === 0) {
        return 'Form created with ' . $new_count . ' field' . ($new_count !== 1 ? 's' : '');
    }

    if ($new_count > $old_count) {
        $diff = $new_count - $old_count;
        return 'Added ' . $diff . ' field' . ($diff !== 1 ? 's' : '') . ' (' . $old_count . ' → ' . $new_count . ')';
    } elseif ($new_count < $old_count) {
        $diff = $old_count - $new_count;
        return 'Removed ' . $diff . ' field' . ($diff !== 1 ? 's' : '') . ' (' . $old_count . ' → ' . $new_count . ')';
    } else {
        return 'Modified field configuration (' . $new_count . ' field' . ($new_count !== 1 ? 's' : '') . ')';
    }
}

// Load existing survey data
$survey_fields = [];
if (file_exists($survey_file)) {
    $survey_fields = json_decode(file_get_contents($survey_file), true);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Form Builder</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
    <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
    <style>
        .form-field {
            cursor: move;
            position: relative;
            padding-left: 30px;
        }
        .form-field::before {
            content: '⋮⋮';
            position: absolute;
            left: 8px;
            top: 50%;
            transform: translateY(-50%);
            color: #666;
            font-size: 16px;
            letter-spacing: -2px;
        }
        .form-field.sortable-ghost {
            opacity: 0.4;
            background: #f0f0f0;
        }
        .form-field.sortable-drag {
            opacity: 1;
            background: #fff;
            box-shadow: 0 5px 15px rgba(0,0,0,0.3);
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <?php include '_sidebar.php'; // Create a _sidebar.php file to avoid repetition, or copy the sidebar HTML here ?>
        
        <main class="content">
            <div class="window">
                <div class="window-title">Form Builder</div>
                <div class="window-content">
                    <fieldset class="controls">
                        <legend>Add New Field</legend>
                        <label for="add-field-type">Field Type:</label>
                        <select id="add-field-type">
                            <option value="title">Title</option>
                            <option value="description">Description Box</option>
                            <option value="hr">Horizontal Line</option>
                            <option value="input">Input Box</option>
                            <option value="textarea">Textarea Box</option>
                            <option value="dropdown">Dropdown Box</option>
                            <option value="checkbox">Checkboxes</option>
                            <option value="radio">Radio Buttons</option>
                        </select>
                        <button id="add-field-btn">Add Field</button>
                    </fieldset>

                    <form id="form-builder-form">
                        <div id="form-builder-area">
                            </div>
                    </form>
                    
                    <button id="save-survey-btn">Save Survey</button>
                    <span id="save-status" style="margin-left: 10px;"></span>
                </div>
            </div>
        </main>
    </div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const builderArea = document.getElementById('form-builder-area');
        const addBtn = document.getElementById('add-field-btn');
        const saveBtn = document.getElementById('save-survey-btn');
        const fieldTypeSelect = document.getElementById('add-field-type');
        const saveStatus = document.getElementById('save-status');

        // Load existing survey fields
        const existingFields = <?php echo json_encode($survey_fields); ?>;
        existingFields.forEach(field => renderField(field));

        // Initialize SortableJS for drag and drop reordering
        const sortable = Sortable.create(builderArea, {
            animation: 150,
            ghostClass: 'sortable-ghost',
            dragClass: 'sortable-drag',
            handle: '.form-field',
            onEnd: function(evt) {
                // Optional: Show a message that the order has changed
                saveStatus.textContent = 'Order changed - click Save to persist';
                setTimeout(() => saveStatus.textContent = '', 3000);
            }
        });

        addBtn.addEventListener('click', function() {
            const type = fieldTypeSelect.value;
            renderField({ type: type, label: '', name: '', text: '', options: [] });
        });

        saveBtn.addEventListener('click', function() {
            saveSurvey();
        });

        function renderField(fieldData) {
            const fieldId = 'field-' + Date.now();
            const div = document.createElement('div');
            div.className = 'form-field';
            div.setAttribute('data-type', fieldData.type);
            
            let html = `<strong>${fieldData.type.toUpperCase()}</strong>`;
            
            if (fieldData.type === 'title') {
                html += `<div><label>Title Text:</label><input type="text" class="field-label" value="${fieldData.label || ''}"></div>`;
            } else if (fieldData.type === 'description') {
                html += `<div><label>Description Text:</label><textarea class="field-text">${fieldData.text || ''}</textarea></div>`;
            } else if (fieldData.type === 'hr') {
                html += `<div style="color: #666; font-style: italic;">Horizontal divider line (no configuration needed)</div>`;
            } else {
                html += `<div><label>Label:</label><input type="text" class="field-label" value="${fieldData.label || ''}"></div>`;
                html += `<div><label>Name (unique key):</label><input type="text" class="field-name" value="${fieldData.name || ''}"></div>`;
                
                if (fieldData.type === 'dropdown' || fieldData.type === 'checkbox' || fieldData.type === 'radio') {
                    html += `<div class="options-area">
                                <label>Options (one per line):</label>
                                <textarea class="field-options">${(fieldData.options || []).join('\n')}</textarea>
                             </div>`;
                }
            }
            
            html += `<div class="field-controls">
                        <button type="button" class="remove-btn">Remove</button>
                     </div>`;
            
            div.innerHTML = html;
            builderArea.appendChild(div);

            div.querySelector('.remove-btn').addEventListener('click', () => {
                div.remove();
            });
        }

        function saveSurvey() {
            const fields = [];
            const fieldElements = builderArea.querySelectorAll('.form-field');
            
            fieldElements.forEach(el => {
                const type = el.getAttribute('data-type');
                const field = { type: type };

                if (type === 'title') {
                    field.label = el.querySelector('.field-label').value;
                } else if (type === 'description') {
                    field.text = el.querySelector('.field-text').value;
                } else if (type === 'hr') {
                    // Horizontal line has no configuration
                } else {
                    field.label = el.querySelector('.field-label').value;
                    field.name = el.querySelector('.field-name').value;
                    if (type === 'dropdown' || type === 'checkbox' || type === 'radio') {
                        field.options = el.querySelector('.field-options').value.split('\n').filter(opt => opt.trim() !== '');
                    }
                }
                fields.push(field);
            });

            saveStatus.textContent = 'Saving...';
            fetch('form-builder.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(fields)
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    saveStatus.textContent = 'Saved successfully!';
                } else {
                    saveStatus.textContent = 'Error: ' + data.message;
                }
                setTimeout(() => saveStatus.textContent = '', 3000);
            })
            .catch(error => {
                saveStatus.textContent = 'Save failed. Check console.';
                console.error('Error:', error);
            });
        }
    });

    // Modal mode detection and parent notification
    (function() {
        const urlParams = new URLSearchParams(window.location.search);
        const isModal = urlParams.get('modal') === '1';

        if (isModal) {
            // Hide "Return to BookGram" link in modal mode
            const returnLink = document.querySelector('a[href="/"]');
            if (returnLink && returnLink.textContent.includes('Return to BookGram')) {
                returnLink.style.display = 'none';
            }

            // Add modal styling adjustments
            document.body.style.margin = '0';
            document.body.style.padding = '0';
            document.body.style.background = '#008080';
        }
    })();
</script>

</body>
</html>
