<?php
/**
 * ChatGram Setup - Initial Instance Configuration
 * Runs when a new chat instance is created
 */

session_start();

// Authenticate
if (isset($_GET['bga_token'])) {
    require_once dirname(__DIR__) . '/app/core/plugin_auth.php';
    $auth = bookgram_auth_check();
    if (!$auth || !bookgram_is_editor()) {
        die('Authentication Required');
    }
}

// Get instance context
$instance_path = $_ENV['BOOKGRAM_INSTANCE_PATH'] ?? null;
$instance_id = $_ENV['BOOKGRAM_INSTANCE_ID'] ?? null;

if (!$instance_path || !$instance_id) {
    die('Instance context not available');
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';

// Check if already configured
if (file_exists($config_file)) {
    // Redirect to admin
    header('Location: admin/index.php?bga_token=' . urlencode($_GET['bga_token'] ?? ''));
    exit;
}

// Load system config for defaults
$system_config_file = dirname(__DIR__) . '/app/storage/config.json';
$system_config = file_exists($system_config_file) ? json_decode(file_get_contents($system_config_file), true) : [];
$chatgram_defaults = $system_config['chatgram'] ?? [
    'default_access_mode' => 'login_only',
    'allow_private_messaging' => true
];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Create data directory
    if (!is_dir($data_dir)) {
        mkdir($data_dir, 0755, true);
    }

    // Create initial config
    $config = [
        'instance_id' => $instance_id,
        'created_at' => date('Y-m-d H:i:s'),
        'created_by' => $_SESSION['bookgram_user'] ?? 'unknown',
        'chat_name' => trim($_POST['chat_name']) ?: 'Chat Room',
        'welcome_message' => trim($_POST['welcome_message']) ?: 'Welcome to the chat!',
        'access_mode' => $_POST['access_mode'] ?? $chatgram_defaults['default_access_mode'],
        'allow_private_messages' => isset($_POST['allow_private_messages'])
    ];

    // Save config
    file_put_contents($config_file, json_encode($config, JSON_PRETTY_PRINT));

    // Initialize data files
    file_put_contents($data_dir . 'messages.json', '[]');
    file_put_contents($data_dir . 'private_messages.json', '[]');
    file_put_contents($data_dir . 'users_online.json', '[]');
    file_put_contents($data_dir . 'guest_sessions.json', '{}');
    file_put_contents($data_dir . 'rate_limits.json', '{}');

    // Redirect to admin
    header('Location: admin/index.php?bga_token=' . urlencode($_GET['bga_token'] ?? ''));
    exit;
}

// Load core functions if not already loaded
if (!function_exists('e')) {
    require_once dirname(__DIR__, 2) . '/app/core/functions.php';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ChatGram Setup</title>
    <link rel="stylesheet" href="/public/css/style.css">
</head>
<body style="background: #008080; padding: 20px;">
    <div class="window" style="max-width: 600px; margin: 0 auto;">
        <div class="window-title-bar">
            <span class="title">ChatGram Setup</span>
        </div>
        <div class="window-content">
            <h2>Configure Your Chat Room</h2>
            <p>Set up your ChatGram instance. You can change these settings later from the admin panel.</p>

            <form method="POST">
                <div style="margin-bottom: 15px;">
                    <label for="chat_name"><strong>Chat Room Name:</strong></label><br>
                    <input type="text" id="chat_name" name="chat_name" value="Chat Room" required style="width: 100%; max-width: 400px;">
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="welcome_message"><strong>Welcome Message:</strong></label><br>
                    <textarea id="welcome_message" name="welcome_message" rows="3" style="width: 100%; max-width: 400px;">Welcome to the chat!</textarea>
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="access_mode"><strong>Access Mode:</strong></label><br>
                    <select id="access_mode" name="access_mode" style="width: 100%; max-width: 400px;">
                        <option value="login_only" <?php echo $chatgram_defaults['default_access_mode'] === 'login_only' ? 'selected' : ''; ?>>Login Only</option>
                        <option value="login_and_guest" <?php echo $chatgram_defaults['default_access_mode'] === 'login_and_guest' ? 'selected' : ''; ?>>Login + Guest Access</option>
                    </select>
                    <small>Guests can use custom names to chat anonymously.</small>
                </div>

                <div style="margin-bottom: 20px;">
                    <label>
                        <input type="checkbox" name="allow_private_messages" checked>
                        <strong>Allow Private Messaging</strong>
                    </label><br>
                    <small>Users can send private 1-on-1 messages.</small>
                </div>

                <button type="submit" style="padding: 10px 20px;">Create Chat Room</button>
            </form>
        </div>
    </div>
</body>
</html>
