<?php
/**
 * RateLimiter - Prevents spam by limiting message rate per user
 */

class RateLimiter {
    private $storage;
    private $max_per_minute;

    public function __construct($storage, $max_per_minute = 10) {
        $this->storage = $storage;
        $this->max_per_minute = $max_per_minute;
    }

    /**
     * Check if user can post (hasn't exceeded rate limit)
     * Returns true if allowed, false if rate limited
     */
    public function checkLimit($user_id) {
        $limits = $this->storage->loadRateLimits();
        $now = time();

        // Clean old entries (older than 60 seconds)
        foreach ($limits as $uid => $data) {
            if ($now - $data['first_message'] > 60) {
                unset($limits[$uid]);
            }
        }

        // Check current user
        if (!isset($limits[$user_id])) {
            // First message in this window
            $limits[$user_id] = [
                'count' => 1,
                'first_message' => $now
            ];
            $this->storage->saveRateLimits($limits);
            return true;
        }

        // Within same minute window
        if ($now - $limits[$user_id]['first_message'] < 60) {
            if ($limits[$user_id]['count'] >= $this->max_per_minute) {
                // Rate limit exceeded
                return false;
            }
            $limits[$user_id]['count']++;
        } else {
            // New minute window
            $limits[$user_id] = [
                'count' => 1,
                'first_message' => $now
            ];
        }

        $this->storage->saveRateLimits($limits);
        return true;
    }

    /**
     * Get remaining messages in current window
     */
    public function getRemainingMessages($user_id) {
        $limits = $this->storage->loadRateLimits();
        $now = time();

        if (!isset($limits[$user_id])) {
            return $this->max_per_minute;
        }

        // Check if window expired
        if ($now - $limits[$user_id]['first_message'] >= 60) {
            return $this->max_per_minute;
        }

        $used = $limits[$user_id]['count'];
        return max(0, $this->max_per_minute - $used);
    }

    /**
     * Get seconds until rate limit resets
     */
    public function getResetTime($user_id) {
        $limits = $this->storage->loadRateLimits();
        $now = time();

        if (!isset($limits[$user_id])) {
            return 0;
        }

        $elapsed = $now - $limits[$user_id]['first_message'];
        if ($elapsed >= 60) {
            return 0;
        }

        return 60 - $elapsed;
    }
}
