<?php
/**
 * GuestManager - Handles guest user sessions and validation
 */

class GuestManager {
    /**
     * Create a new guest session
     */
    public static function createGuestSession($display_name, $instance_id) {
        // Generate unique guest ID
        $guest_id = 'guest_' . bin2hex(random_bytes(8));

        // Store in session
        $_SESSION['chatgram_guest_id'] = $guest_id;
        $_SESSION['chatgram_guest_name'] = $display_name;
        $_SESSION['chatgram_instance_id'] = $instance_id;

        return [
            'guest_id' => $guest_id,
            'display_name' => $display_name,
            'session_id' => session_id()
        ];
    }

    /**
     * Get current guest info from session
     */
    public static function getGuestInfo() {
        if (!isset($_SESSION['chatgram_guest_id'])) {
            return null;
        }

        return [
            'guest_id' => $_SESSION['chatgram_guest_id'],
            'display_name' => $_SESSION['chatgram_guest_name'] ?? 'Anonymous',
            'session_id' => session_id()
        ];
    }

    /**
     * Validate guest name
     * Checks against logged-in usernames and sanitizes input
     */
    public static function validateGuestName($name) {
        // Load Bookgram users if available
        $users_file = dirname(__DIR__, 2) . '/app/storage/users.json';
        if (file_exists($users_file)) {
            $users = json_decode(file_get_contents($users_file), true);

            // Check for impersonation (case-insensitive)
            foreach ($users as $username => $user_data) {
                if (strcasecmp($username, $name) === 0) {
                    return false; // Name matches a registered user
                }
            }
        }

        // Sanitize name
        $name = trim($name);

        // Check length
        if (strlen($name) < 1 || strlen($name) > 30) {
            return false;
        }

        // Remove dangerous characters
        $name = preg_replace('/[<>"\'&]/', '', $name);

        // Check if still valid after sanitization
        if (strlen($name) < 1) {
            return false;
        }

        return $name;
    }

    /**
     * Check if guest name is available
     */
    public static function isGuestNameAvailable($name) {
        return self::validateGuestName($name) !== false;
    }

    /**
     * Generate a random guest name
     */
    public static function generateGuestName() {
        $adjectives = ['Happy', 'Clever', 'Bright', 'Swift', 'Quiet', 'Bold', 'Kind', 'Wise', 'Cool', 'Neat'];
        $nouns = ['Visitor', 'Guest', 'User', 'Friend', 'Chatter', 'Talker', 'Person', 'Buddy', 'Pal', 'Member'];

        $adjective = $adjectives[array_rand($adjectives)];
        $noun = $nouns[array_rand($nouns)];
        $number = rand(100, 999);

        return $adjective . $noun . $number;
    }

    /**
     * Clear guest session
     */
    public static function clearGuestSession() {
        unset($_SESSION['chatgram_guest_id']);
        unset($_SESSION['chatgram_guest_name']);
        unset($_SESSION['chatgram_instance_id']);
    }

    /**
     * Check if current user is a guest
     */
    public static function isGuest() {
        return isset($_SESSION['chatgram_guest_id']);
    }
}
