<?php
/**
 * ChatStorage - File-based storage handler for ChatGram
 * Handles all data persistence with proper file locking
 */

class ChatStorage {
    private $instance_path;
    private $data_dir;

    public function __construct($instance_path) {
        $this->instance_path = $instance_path;
        $this->data_dir = $instance_path . '/data/';

        // Ensure data directory exists
        if (!is_dir($this->data_dir)) {
            mkdir($this->data_dir, 0755, true);
        }
    }

    /**
     * Load JSON data from file with shared lock
     */
    private function loadJSON($filename, $default = []) {
        $file = $this->data_dir . $filename;

        if (!file_exists($file)) {
            return $default;
        }

        $fp = fopen($file, 'r');
        if (!$fp) {
            return $default;
        }

        // Shared lock for reading
        if (flock($fp, LOCK_SH)) {
            $size = filesize($file);
            $content = $size > 0 ? fread($fp, $size) : '[]';
            flock($fp, LOCK_UN);
            fclose($fp);

            $data = json_decode($content, true);
            return $data !== null ? $data : $default;
        }

        fclose($fp);
        return $default;
    }

    /**
     * Save JSON data to file with exclusive lock
     */
    private function saveJSON($filename, $data) {
        $file = $this->data_dir . $filename;

        $fp = fopen($file, 'c+');
        if (!$fp) {
            return false;
        }

        // Exclusive lock for writing
        if (flock($fp, LOCK_EX)) {
            ftruncate($fp, 0);
            rewind($fp);
            $json = json_encode($data, JSON_PRETTY_PRINT);
            fwrite($fp, $json);
            fflush($fp);
            flock($fp, LOCK_UN);
            fclose($fp);
            return true;
        }

        fclose($fp);
        return false;
    }

    // ==================== MESSAGES ====================

    /**
     * Load all public messages
     */
    public function loadMessages() {
        return $this->loadJSON('messages.json', []);
    }

    /**
     * Save public messages
     */
    public function saveMessages($messages) {
        return $this->saveJSON('messages.json', $messages);
    }

    /**
     * Add a new public message
     */
    public function addMessage($author, $author_type, $display_name, $avatar_url, $role, $content) {
        $messages = $this->loadMessages();

        $message = [
            'id' => 'msg_' . bin2hex(random_bytes(8)),
            'type' => 'public',
            'author' => $author,
            'author_type' => $author_type,
            'display_name' => $display_name,
            'avatar_url' => $avatar_url,
            'role' => $role,
            'content' => $content,
            'timestamp' => time(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        $messages[] = $message;
        $this->saveMessages($messages);

        return $message;
    }

    /**
     * Get messages since timestamp
     */
    public function getMessagesSince($since_timestamp = 0, $limit = 50) {
        $messages = $this->loadMessages();

        $filtered = array_filter($messages, function($msg) use ($since_timestamp) {
            return $msg['timestamp'] > $since_timestamp;
        });

        // Sort by timestamp ascending
        usort($filtered, function($a, $b) {
            return $a['timestamp'] - $b['timestamp'];
        });

        // Limit results
        if (count($filtered) > $limit) {
            $filtered = array_slice($filtered, -$limit);
        }

        return array_values($filtered);
    }

    /**
     * Delete a message by ID
     */
    public function deleteMessage($message_id) {
        $messages = $this->loadMessages();

        $filtered = array_filter($messages, function($msg) use ($message_id) {
            return $msg['id'] !== $message_id;
        });

        return $this->saveMessages(array_values($filtered));
    }

    // ==================== PRIVATE MESSAGES ====================

    /**
     * Load all private messages
     */
    public function loadPrivateMessages() {
        return $this->loadJSON('private_messages.json', []);
    }

    /**
     * Save private messages
     */
    public function savePrivateMessages($messages) {
        return $this->saveJSON('private_messages.json', $messages);
    }

    /**
     * Add a private message
     */
    public function addPrivateMessage($conversation_id, $from_user, $from_type, $to_user, $to_type, $content) {
        $messages = $this->loadPrivateMessages();

        $message = [
            'id' => 'pm_' . bin2hex(random_bytes(8)),
            'conversation_id' => $conversation_id,
            'from_user' => $from_user,
            'from_type' => $from_type,
            'to_user' => $to_user,
            'to_type' => $to_type,
            'content' => $content,
            'timestamp' => time(),
            'created_at' => date('Y-m-d H:i:s'),
            'read' => false
        ];

        $messages[] = $message;
        $this->savePrivateMessages($messages);

        return $message;
    }

    /**
     * Get private messages for conversation since timestamp
     */
    public function getPrivateMessagesSince($conversation_id, $since_timestamp = 0) {
        $messages = $this->loadPrivateMessages();

        $filtered = array_filter($messages, function($msg) use ($conversation_id, $since_timestamp) {
            return $msg['conversation_id'] === $conversation_id && $msg['timestamp'] > $since_timestamp;
        });

        // Sort by timestamp ascending
        usort($filtered, function($a, $b) {
            return $a['timestamp'] - $b['timestamp'];
        });

        return array_values($filtered);
    }

    /**
     * Mark messages as read
     */
    public function markMessagesAsRead($conversation_id, $user_id, $user_type) {
        $messages = $this->loadPrivateMessages();
        $modified = false;

        foreach ($messages as &$msg) {
            if ($msg['conversation_id'] === $conversation_id &&
                $msg['to_user'] === $user_id &&
                $msg['to_type'] === $user_type &&
                !$msg['read']) {
                $msg['read'] = true;
                $modified = true;
            }
        }

        if ($modified) {
            $this->savePrivateMessages($messages);
        }

        return $modified;
    }

    /**
     * Get unread count for conversation
     */
    public function getUnreadCount($conversation_id, $user_id, $user_type) {
        $messages = $this->loadPrivateMessages();

        $count = 0;
        foreach ($messages as $msg) {
            if ($msg['conversation_id'] === $conversation_id &&
                $msg['to_user'] === $user_id &&
                $msg['to_type'] === $user_type &&
                !$msg['read']) {
                $count++;
            }
        }

        return $count;
    }

    // ==================== ONLINE USERS ====================

    /**
     * Load online users
     */
    public function loadOnlineUsers() {
        return $this->loadJSON('users_online.json', []);
    }

    /**
     * Save online users
     */
    public function saveOnlineUsers($users) {
        return $this->saveJSON('users_online.json', array_values($users));
    }

    /**
     * Update user heartbeat
     */
    public function updateUserHeartbeat($user_id, $user_type, $display_name, $avatar_url, $role) {
        $users = $this->loadOnlineUsers();
        $now = time();
        $found = false;

        // Update existing user
        foreach ($users as &$user) {
            if ($user['user_id'] === $user_id && $user['user_type'] === $user_type) {
                $user['last_heartbeat'] = $now;
                $user['display_name'] = $display_name;
                $user['avatar_url'] = $avatar_url;
                $user['role'] = $role;
                $found = true;
                break;
            }
        }

        // Add new user
        if (!$found) {
            $users[] = [
                'user_id' => $user_id,
                'user_type' => $user_type,
                'display_name' => $display_name,
                'avatar_url' => $avatar_url,
                'role' => $role,
                'last_heartbeat' => $now,
                'joined_at' => $now
            ];
        }

        // Remove inactive users (no heartbeat for 30 seconds)
        $users = array_filter($users, function($user) use ($now) {
            return ($now - $user['last_heartbeat']) < 30;
        });

        $this->saveOnlineUsers($users);

        return true;
    }

    /**
     * Get online users count
     */
    public function getOnlineUsersCount() {
        $users = $this->loadOnlineUsers();
        return count($users);
    }

    // ==================== GUEST SESSIONS ====================

    /**
     * Load guest sessions
     */
    public function loadGuestSessions() {
        return $this->loadJSON('guest_sessions.json', []);
    }

    /**
     * Save guest sessions
     */
    public function saveGuestSessions($sessions) {
        return $this->saveJSON('guest_sessions.json', $sessions);
    }

    /**
     * Add guest session
     */
    public function addGuestSession($guest_id, $display_name, $session_id, $instance_id) {
        $sessions = $this->loadGuestSessions();

        $sessions[$guest_id] = [
            'guest_id' => $guest_id,
            'display_name' => $display_name,
            'session_id' => $session_id,
            'instance_id' => $instance_id,
            'created_at' => time(),
            'last_active' => time(),
            'ip_hash' => hash('sha256', $_SERVER['REMOTE_ADDR'] ?? 'unknown')
        ];

        $this->saveGuestSessions($sessions);

        return $sessions[$guest_id];
    }

    /**
     * Get guest session
     */
    public function getGuestSession($guest_id) {
        $sessions = $this->loadGuestSessions();
        return $sessions[$guest_id] ?? null;
    }

    /**
     * Update guest activity
     */
    public function updateGuestActivity($guest_id) {
        $sessions = $this->loadGuestSessions();

        if (isset($sessions[$guest_id])) {
            $sessions[$guest_id]['last_active'] = time();
            $this->saveGuestSessions($sessions);
            return true;
        }

        return false;
    }

    /**
     * Clean up inactive guest sessions
     */
    public function cleanupInactiveGuests($timeout_minutes = 10) {
        $sessions = $this->loadGuestSessions();
        $cutoff = time() - ($timeout_minutes * 60);

        $active = array_filter($sessions, function($session) use ($cutoff) {
            return $session['last_active'] >= $cutoff;
        });

        if (count($active) < count($sessions)) {
            $this->saveGuestSessions($active);
        }

        return count($sessions) - count($active);
    }

    // ==================== CONFIG ====================

    /**
     * Load instance config
     */
    public function loadConfig() {
        return $this->loadJSON('config.json', []);
    }

    /**
     * Save instance config
     */
    public function saveConfig($config) {
        return $this->saveJSON('config.json', $config);
    }

    // ==================== RATE LIMITS ====================

    /**
     * Load rate limits
     */
    public function loadRateLimits() {
        return $this->loadJSON('rate_limits.json', []);
    }

    /**
     * Save rate limits
     */
    public function saveRateLimits($limits) {
        return $this->saveJSON('rate_limits.json', $limits);
    }
}
